<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model as Eloquent;

class Quotation extends Eloquent
{
    public const QUOTATION_CREATE_ACTIVITY = 'Quotation Created';
    public const QUOTATION_UPDATE_ACTIVITY = 'Quotation Updated';
    public const QUOTATION_DELETE_ACTIVITY = 'Quotation Deleted';
    public const QUOTATION_PATH = '/quotation/list';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'quotation_no', 'customer_id', 'date', 'status_id', 'details', 'sub_total', 'tax_amount', 'gross_total'
    ];

    protected $dates = [
        'date'  => 'datetime:Y-m-d',
    ];

    protected static function booted()
    {
        static::creating(function ($quotation) {
            $quotation->quotation_no = $quotation->generateUniqueIdentity();
        });

        static::updating(function ($quotation) {
			$quotation->date = date('Y-m-d', strtotime($quotation->date) );
        });
    }

    public function status()
    {
        return $this->belongsTo(Category::class, 'status_id');
    }

    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    public function productDetails()
    {
        return $this->hasMany(QuotationProducts::class);
    }

    public function taxes()
    {
        return $this->belongsToMany(Tax::class, null, 'reference_ids', 'tax_ids');
    }

    public function generateUniqueIdentity()
    {
        $data = $this->select(['quotation_no'])->orderBy('created_at', 'desc')->first();
        $prefix = 'Q-';
        if (!empty($data->quotation_no)) {


            $split = explode("-", $data->quotation_no);
            $find = sizeof($split) - 1;
            $last_id = $split[$find];
            $number = intval($last_id) + 1;
            $new_no = sprintf('%06d', $number);
            $uniqID = $prefix . $new_no;
            return $uniqID;
        } else {
            return $prefix . '000001';
        }
    }
}
