<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model as Eloquent;

class Purchase extends Eloquent
{

    public const PURCHASE_CREATE_ACTIVITY = 'Purchase Created';
    public const PURCHASE_UPDATE_ACTIVITY = 'Purchase Updated';
    public const PURCHASE_DELETE_ACTIVITY = 'Purchase Deleted';
    public const PURCHASE_PATH = '/purchase/list';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'purchase_no', 'date', 'status', 'contact_person', 'supplier_id', 'total_amount'
    ];

    protected $dates = [
        'date'  => 'datetime:Y-m-d',
    ];

    protected static function booted()
    {
        static::creating(function ($purchase) {
            $purchase->purchase_no = $purchase->generateUniqueIdentity();
        });

        static::updating(function ($purchase) {
            $purchase->date = date('Y-m-d', strtotime($purchase->date));
        });
    }

    public function productDetails()
    {
        return $this->hasMany(PurchaseProducts::class);
    }

    public function purchaseStatus()
    {
        return $this->belongsTo(Category::class, 'status');
    }

    public function supplier()
    {
        return $this->belongsTo(User::class, 'supplier_id');
    }

    public function stock()
    {
        return $this->hasOne(Stock::class);
    }

    public function generateUniqueIdentity()
    {
        $data = $this->select(['purchase_no'])->orderBy('created_at', 'desc')->first();
        $prefix = 'P-';
        if (!empty($data->purchase_no)) {


            $split = explode("-", $data->purchase_no);
            $find = sizeof($split) - 1;
            $last_id = $split[$find];
            $number = intval($last_id) + 1;
            $new_no = sprintf('%06d', $number);
            $uniqID = $prefix . $new_no;
            return $uniqID;
        } else {
            return $prefix . '000001';
        }
    }
}
