<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model as Eloquent;

class Product extends Eloquent
{
    public const PRODUCT_CREATE_ACTIVITY = 'Product Created';
    public const PRODUCT_UPDATE_ACTIVITY = 'Product Updated';
    public const PRODUCT_DELETE_ACTIVITY = 'Product Deleted';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'product_identity', 'name', 'model_number', 'brand_id', 'category_id', 'price', 'unit_id', 'image', 'description', 'is_archive'
    ];

    protected static function booted()
    {
        static::creating(function ($product) {
            $product->product_identity = $product->generateUniqueIdentity();
            $product->is_archive = 0;
        });

        // Delete all related model data
        static::deleting(function($product) { 
            $product->stock()->each(function($stock) {
               $stock->delete(); 
            });
       });
    }

    public function amc()
    {
        return $this->hasMany(AmcProduct::class,'amc_product');
    }

    public function quotation()
    {
        return $this->hasMany(QuotationProducts::class,'product_id');
    }

    public function sale()
    {
        return $this->hasMany(SaleProducts::class,'product_id');
    }

    public function brand()
    {
        return $this->belongsTo(Category::class, 'brand_id');
    }

    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id');
    }

    public function unit()
    {
        return $this->belongsTo(Category::class, 'unit_id');
    }

    public function stock()
    {
        return $this->hasOne(Stock::class);
    }

    public function generateUniqueIdentity()
    {
        $data = $this->select(['product_identity'])->orderBy('created_at', 'desc')->first();
        $prefix = 'P-';
        if (!empty($data->product_identity)) {


            $split = explode("-", $data->product_identity);
            $find = sizeof($split) - 1;
            $last_id = $split[$find];
            $number = intval($last_id) + 1;
            $new_no = sprintf('%06d', $number);
            $uniqID = $prefix . $new_no;
            return $uniqID;
        } else {
            return $prefix . '000001';
        }
    }
}
