<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Conversation extends Model
{
    protected $fillable = ['sender_id', 'subject', 'attachment', 'message_comment', 'sms_text', 'send_sms', 'send_mail', 'send_by'];

    public function members()
    {
        return $this->hasMany(ConversationMember::class);
    }

    public function messages()
    {
        return $this->hasMany(ConversationMessage::class);
    }

    public function owner()
    {
        return $this->belongsTo(User::class, 'created_by')->withDefault(['full_name' => 'Unknown']);
    }

    protected static function booted()
    {
        static::creating(function ($model) {
            if (! $model->created_by) {
                $model->created_by = auth()->id();
            }
        });
    }

    public function markMessagesAsReadForUser($userId)
    {
        $this->load('messages');

        $this->messages->each(function ($message) use ($userId) {
            if ($message->created_by != $userId) {
                $message->markAsRead($userId);
            }
        });
    }

    public function filteredMessagesForUser($loggedInUserId)
    {
        return $this->messages->transform(function ($message) use ($loggedInUserId) {
            if ($message->deleted_by && in_array($loggedInUserId, $message->deleted_by)) {
                return null;
            }

            return $message;
        })->filter();
    }
}
