<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model as Eloquent;

class Complaint extends Eloquent
{
    public const COMPLAINT_CREATE_ACTIVITY = 'Complaint Created';
    public const COMPLAINT_UPDATE_ACTIVITY = 'Complaint Updated';
    public const COMPLAINT_DELETE_ACTIVITY = 'Complaint Deleted';
    public const COMPLAINT_PATH = '/complaint/list';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'complaint_no', 'customer_id', 'date', 'complaint_time', 'status_id', 'complaint_type_id', 'description', 'product_id', 'assign_to_id', 'assign_date', 'attachment'
    ];

    protected $dates = [
        'date'  => 'datetime:Y-m-d',
        'assign_date'  => 'datetime:Y-m-d',
    ];

    protected static function booted()
    {
        static::creating(function ($complaint) {
            $complaint->complaint_no = $complaint->generateUniqueIdentity();
        });

        static::updating(function ($complaint) {
            $complaint->date = date('Y-m-d', strtotime($complaint->date));
            $complaint->assign_date = date('Y-m-d', strtotime($complaint->assign_date));
        });
    }

    public function customer()
    {
        return $this->belongsTo(User::class, 'customer_id');
    }

    public function status()
    {
        return $this->belongsTo(Category::class, 'status_id');
    }

    public function complaintType()
    {
        return $this->belongsTo(Category::class, 'complaint_type_id');
    }

    public function product()
    {
        return $this->belongsTo(Product::class, 'product_id');
    }

    public function assignTo()
    {
        return $this->belongsTo(User::class, 'assign_to_id');
    }

    public function generateUniqueIdentity()
    {
        $data = $this->select(['complaint_no'])->orderBy('created_at', 'desc')->first();
        $prefix = 'C-';
        if (!empty($data->complaint_no)) {


            $split = explode("-", $data->complaint_no);
            $find = sizeof($split) - 1;
            $last_id = $split[$find];
            $number = intval($last_id) + 1;
            $new_no = sprintf('%06d', $number);
            $uniqID = $prefix . $new_no;
            return $uniqID;
        } else {
            return $prefix . '000001';
        }
    }
}
