<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model as Eloquent;
use Carbon\Carbon;

class Amc extends Eloquent
{
    public const AMC_CREATE_ACTIVITY = 'AMC Created';
    public const AMC_UPDATE_ACTIVITY = 'AMC Updated';
    public const AMC_DELETE_ACTIVITY = 'AMC Deleted';
    public const AMC_PATH = '/amc/list';


    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'amc_no', 'customer', 'date', 'contact_person', 'status', 'employee_status', 'is_employee_status_approved', 'assign_to', 'attachment', 'amc_detail', 'interval', 'number_of_service'
    ];

    protected $dates = [
        'date'  => 'datetime:Y-m-d',
    ];

    protected static function booted()
    {
        static::creating(function ($amc) {
            $amc->amc_no = $amc->generateUniqueIdentity();
            $amc->employee_status = request()->status;
            $amc->is_employee_status_approved = false;
        });

        static::updating(function ($amc) {
            // if($amc->employee_status != $amc->status)
            // {
                // $amc->is_employee_status_approved = false;
            // }
			$amc->date = date('Y-m-d', strtotime($amc->date) );
        });

        // Delete all related model data
        static::deleting(function ($amc) {
            $amc->services()->each(function ($service) {
                $service->delete();
            });

            $amc->products()->each(function ($product) {
                $product->delete();
            });
        });
    }

    public function getNextServiceAttribute()
    {
        return $this->services()->where('service_date', '>=', date('Y-m-d'))->first();
    }

    public function client()
    {
        return $this->belongsTo(User::class, 'customer');
    }

    public function amcStatus()
    {
        return $this->belongsTo(Category::class, 'status');
    }

    public function staffStatus()
    {
        return $this->belongsTo(Category::class, 'employee_status');
    }

    public function employee()
    {
        return $this->belongsTo(User::class, 'assign_to');
    }

    public function services()
    {
        return $this->hasMany(AmcService::class);
    }

    public function products()
    {
        return $this->hasMany(AmcProduct::class);
    }

    public function generateUniqueIdentity()
    {
        $data = $this->select(['amc_no'])->orderBy('created_at', 'desc')->first();
        $prefix = 'AMC-';
        if (!empty($data->amc_no)) {


            $split = explode("-", $data->amc_no);
            $find = sizeof($split) - 1;
            $last_id = $split[$find];
            $number = intval($last_id) + 1;
            $new_no = sprintf('%06d', $number);
            $uniqID = $prefix . $new_no;
            return $uniqID;
        } else {
            return $prefix . '000001';
        }
    }
}
