<?php

namespace App\Listeners;

use App\Events\SendFcmNotification;
use Illuminate\Support\Facades\Http;
use Firebase\JWT\JWT;

class FcmNotificationListener
{
    /**
     * Create the event listener.
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    /**
     * Handle the event.
     *
     * @param SendFcmNotification $event
     * @return void
     */
    public function handle(SendFcmNotification $event)
    {
        $url = 'https://fcm.googleapis.com/v1/projects/amc-app-48ac1/messages:send';
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            \Log::error('Unable to obtain FCM access token. Notifications will not be sent.');
            return;
        }

        foreach ($event->notificationData as $notification) {
            $deviceToken = $notification['deviceToken'];
            $title = $notification['title'];
            $body = $notification['body'];
            if (empty($deviceToken)) {
                // Handle the case when deviceToken is empty or null
                continue;
            }

            $fields = [
                'message' => [
                    'token' => $deviceToken,
                    'notification' => [
                        'title' => $title,
                        'body' => $body,
                    ],
                ],
            ];

            $headers = [
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json; UTF-8',
            ];

            $response = Http::withHeaders($headers)->post($url, $fields);

            if ($response->successful()) {
                // Notification sent successfully
                $responseData = $response->json();
                // dump('Notification sent successfully');
                // dd($responseData);
            } else {
                // Notification sending failed
                // dd($response->body());
            }
        }
    }

    /**
     * Get the access token for Firebase Cloud Messaging.
     *
     * @return string
     */
    private function getAccessToken()
    {
        try {
            $privateKey = str_replace('\n', "\n", env('FCM_PRIVATE_KEY'));
            $clientEmail = env('GOOGLE_CLIENT_EMAIL');
            $tokenUri = 'https://oauth2.googleapis.com/token';
            $scope = 'https://www.googleapis.com/auth/firebase.messaging';

            if (empty($privateKey) || empty($clientEmail)) {
                \Log::error('FCM_PRIVATE_KEY or GOOGLE_CLIENT_EMAIL is missing in .env file');
                return null;
            }

            $jwtToken = JWT::encode([
                'iss' => $clientEmail,
                'scope' => $scope,
                'aud' => $tokenUri,
                'exp' => time() + 3600,
                'iat' => time(),
            ], $privateKey, 'RS256');

            $response = Http::asForm()->post($tokenUri, [
                'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
                'assertion' => $jwtToken,
            ]);

            if ($response->successful()) {
                $responseData = $response->json();
                return $responseData['access_token'] ?? null;
            } else {
                \Log::error('Failed to obtain access token: ' . $response->body());
                return null;
            }
        } catch (\Exception $e) {
            \Log::error('Error in getAccessToken: ' . $e->getMessage());
            return null;
        }
    }
}