<?php

namespace App\Http\Controllers;

use App\Events\SendFcmNotification;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\ActivityLog;
use App\Models\UserDetails;
use App\Models\Notifications;
use App\Http\Controllers\Controller;
use App\Http\Resources\CustomerProfileResource;
use App\Http\Resources\EmployeeProfileResource;
use App\Http\Resources\PermissionResource;
use App\Http\Resources\SupplierProfileResource;
use App\Http\Resources\UserResource;
use App\Http\Resources\UserpluckResource;
use App\Http\Resources\UserDetailsResource;
use App\Mail\Notification;
use App\Mail\Register;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Exceptions\UserNotDefinedException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    /*User List Data show */
    public function index()
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('user_view')) {
                if (auth()->user()->hasPermissionTo('user_own_data')) {
                    $userData = User::role(['customer', 'employee', 'supplier'])->where(['id' => auth()->user()->id])->Orderby('created_at', 'desc')->get();
                } else {
                    $userData = User::role(['customer', 'employee', 'supplier'])->Orderby('created_at', 'desc')->get();
                }
                return UserResource::collection($userData)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*User List By Role */
    public function userByRole(request $request)
    {
        $validator = Validator::make($request->all(), [
            'role' => 'required|in:customer,employee,supplier',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('user_view')) {

                $userData = User::role($request->role)->Orderby('created_at', 'desc')->get();
                return UserResource::collection($userData)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*New User Insert Data */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'first_name' => 'required',
            'last_name' => 'required',
            'gender' => 'required|in:male,female',
            'email' => 'required|email|unique:users',
            'password' => 'required|confirmed|min:8',
            'profile' => 'filled|mimes:jpg,bmp,png,jpeg,svg|max:2048',
            'birth_date' => 'nullable|date',
            'mobile' => 'required|between:6,15',
            'role' => 'required|in:customer,employee,supplier,admin',

            // 'company_name' => 'required_if:role,customer',
            // 'company_contact_person' => 'required_if:role,customer',
            // 'account_number' => 'required_if:role,customer,supplier|between:9,18',
            // 'ifsc_code' => 'required_if:role,customer',
            // 'branch_name' => 'required_if:role,customer,supplier',
            // 'tin_no' => 'required_if:role,customer',
            // 'cst_no' => 'required_if:role,customer',
            // 'pan_no' => 'required_if:role,customer',
            // 'phone' => 'required_if:role,customer|numeric',

            'company_name' => 'nullable',
            'company_contact_person' => 'nullable',
            'account_number' => 'nullable|between:9,18',
            'ifsc_code' => 'nullable',
            'branch_name' => 'nullable',
            'tin_no' => 'nullable',
            'cst_no' => 'nullable',
            'pan_no' => 'nullable',
            'phone' => 'nullable|numeric',

            'address' => 'required',
            'city' => 'required',
            'state' => 'required',
            'pincode' => 'required',

            'contact_person' => 'filled|array',
            'contact_person.*' => 'required',
            'contact_mobile' => 'filled|array',
            'contact_mobile.*' => 'required|between:6,15',
            'contact_designation' => 'filled|array',
            'contact_designation.*' => 'required',

            // 'bank_name' => 'required_if:role,supplier',
            // 'swift_code' => 'required_if:role,supplier',
            // 'branch_code' => 'required_if:role,supplier',
            // 'international_bank_code' => 'required_if:role,supplier',
            // 'national_bank_code' => 'required_if:role,supplier',
            // 'gst' => 'required_if:role,supplier',

            'bank_name' => 'nullable',
            'swift_code' => 'nullable',
            'branch_code' => 'nullable',
            'international_bank_code' => 'nullable',
            'national_bank_code' => 'nullable',
            'gst' => 'nullable',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('user_add')) {

                $user = User::create($request->except('role', 'profile'));

                $user->assignRole($request->role);

                if ($request->has('profile')) {
                    $userProfile = $this->uploadFile($request->profile, 'images/user');
                    if ($userProfile != false) {
                        $user->profile = $userProfile;
                        $user->save();
                    }
                }

                $user->save();

                if ($request->has('contact_person')) {
                    // Customer Contacts
                    foreach ($request->contact_person as $key => $value) {
                        $contacts =  $user->customerContacts()->create([
                            'contact_person' => $value,
                            'mobile' => $request->contact_mobile[$key],
                            'designation' => $request->contact_designation[$key],
                        ]);
                    }
                }

             
                if ($user->getRole() != "supplier") {
                    $systemName = getSetting('system_name');
                    // Get email template content using helper
                    $emailTemplate = getEmailContent('registration');
                    $logo = getSetting('logo');
                    $logo_image = url('storage/' . $logo);
                    $footer_image = url('storage/email_image/footer.png');
                    $support_image = url('storage/email_image/support.png');
                    $service_image = url('storage/email_image/service.png');
                    $history_image = url('storage/email_image/history.png');

                    if ($emailTemplate) {
                        // dump($emailTemplate);
                        $mailformat = $emailTemplate->body;
                        $serch = array('{ user_name }', '{ system_name }', '{ username }', '{ password }', '{ role }');
                        $replace = array($user->full_name, $systemName, $user->email, $request->password, $request->role);
                        $message_content = str_replace($serch, $replace, $mailformat);
                        // Welcome Mail
                        $mailData = [
                            'host' => getHost(),
                            'subject' => $emailTemplate->subject,
                            'content' => $message_content,
                            'logo_image' => $logo_image,
                            'footer_image' => $footer_image,
                            'support_image' => $support_image,
                            'service_image' => $service_image,
                            'history_image' => $history_image
                        ];
                        // dump($request->email);
                        Mail::to($request->email)->send(new Register($mailData));
                        if (Mail::failures()) {
                            throw new \Exception('Failed to send email. Please try again.');
                        }else{
                        // dump('Mail Sent');
                        }
                    }
                }

                $sourceName = auth()->id();
                $user1 = User::find($sourceName);
                $sourceId = $user1 ? $user1->first_name : 'Unknown User'; // Assuming the user ID is the source ID
                $targetName = $user->first_name; // Assuming the target is the user profile
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => User::USER_CREATE_ACTIVITY,
                    'activity_slug' => Str::of( User::USER_CREATE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => $targetName . ' created their profile',
                ];
                ActivityLog::log($data);
                $admin = User::role(['admin'])->get();
                if(auth()->user()->getRole() == 'admin') {
                $notifications = [
                    [
                        'user_id' => $user->id , // Change this to the appropriate user ID
                        'target_id' => $user->id,
                        'updated_by' => $sourceId,
                        'content' => "Your profile has been created as {$user->getRole()}.",
                        'is_read' => 0, // Set as unread (0) by default
                        'path' => User::USER_PATH
                    ],
                ];
                Notifications::send($notifications);
                $notificationData = [
                    [
                    'deviceToken' => $user->device_token,
                    'title' => "New User Created",
                    'body' => "Your profile has been created as {$user->getRole()}."
                    ]
                ];
                event(new SendFcmNotification($notificationData));
                }
                else {
                    $notifications = [
                        [
                            'user_id' => $user->id , // Change this to the appropriate user ID
                            'target_id' => $user->id,
                            'updated_by' => $sourceId,
                            'content' => "Your profile has been created as {$user->getRole()}.",
                            'is_read' => 0, // Set as unread (0) by default
                            'path' => User::USER_PATH
                        ],
                        [
                            'user_id' => $admin->first()->id , // Change this to the appropriate user ID
                            'target_id' => $user->id,
                            'updated_by' => $sourceId,
                            'content' => "A new {$user->getRole()} {$user->first_name} has been added.",
                            'is_read' => 0, // Set as unread (0) by default
                            'path' => User::USER_PATH
                        ],
                    ];
                    Notifications::send($notifications);
                    $notificationData = [
                        [
                            'deviceToken' => $user->device_token,
                            'title' => "New User Created",
                            'body' => "Your profile has been created as {$user->getRole()}."
                        ],
                        [
                            'deviceToken' => $admin->first()->device_token,
                            'title' => "New User Created",
                            'body' => "A new {$user->getRole()} {$user->first_name} has been added."
                        ]
                        ];
                        event(new SendFcmNotification($notificationData));
                };
                $this->_response['data'] = '';
                $this->setResponse(false, 'User created successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Update User Data */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'first_name' => 'required',
            'last_name' => 'required',
            'gender' => 'required|in:male,female',
            'password' => 'filled|confirmed|min:8',
            'profile' => 'filled|mimes:jpg,bmp,png,jpeg,svg|max:2048',
            'birth_date' => 'nullable|date',
            'mobile' => 'required|between:6,15',
            'role' => 'required|in:customer,employee,supplier,admin',

            // 'company_name' => 'required_if:role,customer',
            // 'company_contact_person' => 'required_if:role,customer',
            // 'account_number' => 'required_if:role,customer,supplier|between:9,18',
            // 'ifsc_code' => 'required_if:role,customer',
            // 'branch_name' => 'required_if:role,customer,supplier',
            // 'tin_no' => 'required_if:role,customer',
            // 'cst_no' => 'required_if:role,customer',
            // 'pan_no' => 'required_if:role,customer',
            // 'phone' => 'required_if:role,customer|numeric',

            'company_name' => 'nullable',
            'company_contact_person' => 'nullable',
            'account_number' => 'nullable|between:9,18',
            'ifsc_code' => 'nullable',
            'branch_name' => 'nullable',
            'tin_no' => 'nullable',
            'cst_no' => 'nullable',
            'pan_no' => 'nullable',
            'phone' => 'nullable|numeric',

            'address' => 'required',
            'city' => 'required',
            'state' => 'required',
            'pincode' => 'required',

            'contact_person' => 'filled|array',
            'contact_person.*' => 'required',
            'contact_mobile' => 'filled|array',
            'contact_mobile.*' => 'required|between:6,15',
            'contact_designation' => 'filled|array',
            'contact_designation.*' => 'required',

            // 'bank_name' => 'required_if:role,supplier',
            // 'swift_code' => 'required_if:role,supplier',
            // 'branch_code' => 'required_if:role,supplier',
            // 'international_bank_code' => 'required_if:role,supplier',
            // 'national_bank_code' => 'required_if:role,supplier',
            // 'gst' => 'required_if:role,supplier',

            'bank_name' => 'nullable',
            'swift_code' => 'nullable',
            'branch_code' => 'nullable',
            'international_bank_code' => 'nullable',
            'national_bank_code' => 'nullable',
            'gst' => 'nullable',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('user_edit')) {

                $user = User::find(auth()->id());
                $user->update($request->except(['role', 'profile', 'email']));

                if ($request->has('profile')) {

                    $userProfile = $this->uploadFile($request->profile, 'images/user');
                    if ($userProfile != false) {
                        $user->profile = $userProfile;
                        $user->save();
                    }
                }

                $user->save();

                if ($request->has('contact_person')) {
                    // Customer Contacts
                    $user->customerContacts()->delete();

                    // Customer Contacts
                    foreach ($request->contact_person as $key => $value) {
                        $contacts =  $user->customerContacts()->create([
                            'contact_person' => $value,
                            'mobile' => $request->contact_mobile[$key],
                            'designation' => $request->contact_designation[$key],
                        ]);
                    }
                }

                $sourceName = auth()->id();
                    $user1 = User::find($sourceName);
                    $sourceId = $user1 ? $user1->first_name : 'Unknown User'; // Assuming the user ID is the source ID
                $targetName = $user->first_name; // Assuming the target is the user profile
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => User::USER_UPDATE_ACTIVITY,
                    'activity_slug' => Str::of( User::USER_UPDATE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => $targetName . ' updated their profile',
                ];
                ActivityLog::log($data);
                $admin = User::role(['admin'])->get();
                if(auth()->user()->getRole() == 'admin') {
                    $notifications = [
                        [
                            'user_id' => $user->id , // Change this to the appropriate user ID
                            'target_id' => $user->id,
                            'updated_by' => $sourceId,
                            'content' => "Your profile has been updated.",
                            'is_read' => 0, // Set as unread (0) by default
                            'path' => User::USER_PATH
                        ],
                    ];
                    Notifications::send($notifications);
                    $notificationData = [
                        [
                            'deviceToken' => $user->device_token,
                            'title' => 'Profile Updated',
                            'body' => 'Your profile has been updated.',
                        ]
                    ];
                    event(new SendFcmNotification($notificationData));
                    }
                    else {
                        $notifications = [
                            [
                                'user_id' => $user->id , // Change this to the appropriate user ID
                                'target_id' => $user->id,
                                'updated_by' => $sourceId,
                                'content' => "Your profile has been updated.",
                                'is_read' => 0, // Set as unread (0) by default
                                'path' => User::USER_PATH
                            ],
                            [
                                'user_id' => $admin->first()->id , // Change this to the appropriate user ID
                                'target_id' => $user->id,
                                'updated_by' => $sourceId,
                                'content' => "User profile of {$user->first_name} has been updated.",
                                'is_read' => 0, // Set as unread (0) by default
                                'path' => User::USER_PATH
                            ],
                        ];
                        Notifications::send($notifications);
                        $notificationData = [
                            [
                                'deviceToken' => $user->device_token,
                                'title' => 'Profile Updated',
                                'body' => 'Your profile has been updated.',
                            ],
                            [
                                'deviceToken' => $admin->first()->device_token,
                                'title' => 'Profile Updated',
                                'body' => "User profile of {$user->first_name} has been updated.",
                            ]
                        ];
                        event(new SendFcmNotification($notificationData));
                    };
                return (new UserResource($user))->additional(["error" => false, "message" => 'Profile updated successfully.']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Update User Data */
    public function updateOtherUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'first_name' => 'required',
            'last_name' => 'required',
            'gender' => 'required|in:male,female',
            'password' => 'filled|confirmed|min:8',
            'profile' => 'filled|mimes:jpg,bmp,png,jpeg,svg|max:2048',
            'birth_date' => 'nullable|date',
            'mobile' => 'required|between:6,15',
            'role' => 'required|in:customer,employee,supplier,admin',

            // 'company_name' => 'required_if:role,customer',
            // 'company_contact_person' => 'required_if:role,customer',
            // 'account_number' => 'required_if:role,customer,supplier|between:9,18',
            // 'ifsc_code' => 'required_if:role,customer',
            // 'branch_name' => 'required_if:role,customer,supplier',
            // 'tin_no' => 'required_if:role,customer',
            // 'cst_no' => 'required_if:role,customer',
            // 'pan_no' => 'required_if:role,customer',
            // 'phone' => 'required_if:role,customer|numeric',

            'company_name' => 'nullable',
            'company_contact_person' => 'nullable',
            'account_number' => 'nullable|between:9,18',
            'ifsc_code' => 'nullable',
            'branch_name' => 'nullable',
            'tin_no' => 'nullable',
            'cst_no' => 'nullable',
            'pan_no' => 'nullable',
            'phone' => 'nullable|numeric',

            'address' => 'required',
            'city' => 'required',
            'state' => 'required',
            'pincode' => 'required',

            'contact_person' => 'filled|array',
            'contact_person.*' => 'required',
            'contact_mobile' => 'filled|array',
            'contact_mobile.*' => 'required|between:6,15',
            'contact_designation' => 'filled|array',
            'contact_designation.*' => 'required',

            // 'bank_name' => 'required_if:role,supplier',
            // 'swift_code' => 'required_if:role,supplier',
            // 'branch_code' => 'required_if:role,supplier',
            // 'international_bank_code' => 'required_if:role,supplier',
            // 'national_bank_code' => 'required_if:role,supplier',
            // 'gst' => 'required_if:role,supplier',

            'bank_name' => 'nullable',
            'swift_code' => 'nullable',
            'branch_code' => 'nullable',
            'international_bank_code' => 'nullable',
            'national_bank_code' => 'nullable',
            'gst' => 'nullable',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('user_edit')) {

                $user = User::find($request->user_id);
                $user->update($request->except(['role', 'profile', 'email']));

                if ($request->has('profile')) {

                    $userProfile = $this->uploadFile($request->profile, 'images/user');
                    if ($userProfile != false) {
                        $user->profile = $userProfile;
                        $user->save();
                    }
                }

                if ($request->has('contact_person')) {
                    // Customer Contacts
                    $user->customerContacts()->delete();

                    // Customer Contacts
                    foreach ($request->contact_person as $key => $value) {
                        $contacts =  $user->customerContacts()->create([
                            'contact_person' => $value,
                            'mobile' => $request->contact_mobile[$key],
                            'designation' => $request->contact_designation[$key],
                        ]);
                    }
                }

                $sourceName = auth()->id();
                    $user1 = User::find($sourceName);
                    $sourceId = $user1 ? $user1->first_name : 'Unknown User'; // Assuming the user ID is the source ID
                $targetName = $user->first_name; // Assuming the target is the user profile
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => User::USER_UPDATE_ACTIVITY,
                    'activity_slug' => Str::of( User::USER_UPDATE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => $targetName . ' updated their profile',
                ];
                ActivityLog::log($data);
                $admin = User::role(['admin'])->get();
                if(auth()->user()->getRole() == 'admin') {
                    $notifications = [
                        [
                            'user_id' => $user->id , // Change this to the appropriate user ID
                            'target_id' => $user->id,
                            'updated_by' => $sourceId,
                            'content' => "Your profile has been updated.",
                            'is_read' => 0, // Set as unread (0) by default
                            'path' => User::USER_PATH
                        ],
                    ];
                    Notifications::send($notifications);
                    $notificationData = [
                        [
                            'deviceToken' => $user->device_token,
                            'title' => 'Profile Updated',
                            'body' => 'Your profile has been updated.',
                        ]
                        ];
                        event(new SendFcmNotification($notificationData));
                    }
                    else {
                        $notifications = [
                            [
                                'user_id' => $user->id , // Change this to the appropriate user ID
                                'target_id' => $user->id,
                                'updated_by' => $sourceId,
                                'content' => "Your profile has been updated.",
                                'is_read' => 0, // Set as unread (0) by default
                                'path' => User::USER_PATH
                            ],
                            [
                                'user_id' => $admin->first()->id , // Change this to the appropriate user ID
                                'target_id' => $user->id,
                                'updated_by' => $sourceId,
                                'content' => "User profile of {$user->first_name} has been updated.",
                                'is_read' => 0, // Set as unread (0) by default
                                'path' => User::USER_PATH
                            ],
                        ];
                        Notifications::send($notifications);
                        $notificationData = [
                            [
                                'deviceToken' => $user->device_token,
                                'title' => 'Profile Updated',
                                'body' => 'Your profile has been updated.',
                            ],
                            [
                                'deviceToken' => $admin->first()->device_token,
                                'title' => 'Profile Updated',
                                'body' => "User profile of {$user->first_name} has been updated.",
                            ]
                        ];
                        event(new SendFcmNotification($notificationData));
                    };
                return (new UserResource($user))->additional(["error" => false, "message" => 'User updated successfully.']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete User Data*/
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('user_delete')) {

                $userDelete =  User::find($request->user_id);
                if ($userDelete) {
                    $userDelete->customerContacts()->delete();
                    $userDelete->delete();

                    $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                    $targetName = $userDelete->first_name;

                    $data = [
                    'updated_by' => $sourceId,
                    'activity' => User::USER_DELETE_ACTIVITY,
                    'activity_slug' => Str::of( User::USER_DELETE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => $targetName . ' deleted their profile',
                    ];
                ActivityLog::log($data);

                    $this->setResponse(false, 'User deleted successfully.');
                    return response()->json($this->_response);
                }
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete multiple User Data*/
    public function deleteMultiUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_ids' => 'required|array',
            'user_ids.*' => 'required|exists:users,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('user_delete')) {

                foreach ($request->user_ids as $id) {
                    $userDelete =  User::find($id);
                    if ($userDelete) {
                        $userDelete->customerContacts()->delete();
                        $userDelete->delete();
                    }
                }

                $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                $targetName = $userDelete->first_name;
                $activityData = [
                    'updated_by' => $sourceId,
                    'activity' => User::USER_DELETE_ACTIVITY,
                    'activity_slug' => Str::of( User::USER_DELETE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => "{$targetName} deleted their profile",
                ];
                ActivityLog::log($activityData);
                $this->setResponse(false, 'Users deleted successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /* All User Profile List Api  (list)*/
    public function allProfiles()
    {
        try {
            $userProfileData = UserDetails::all();
            return UserDetailsResource::collection($userProfileData)->additional(["error" => false, "message" => 'Here is all users Profile data']);
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /* Update User Profile Api  (Update)*/

    public function updateUserPrfile(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required',
            'gender' => 'required',
            'phone' => 'required|numeric|min:10',
            'dob' => 'required|date',
            'country' => 'required',
            'nationality' => 'required',
            'aadhar_id' => 'required|numeric|min:16',
            'area' => 'required',
            'city' => 'required',
            'district' => 'required',
            'state' => 'required',
            'geo_location' => 'filled',
            'marital_status' => 'filled',/*'boolean'*/
            'education_details' => 'required',
            'languages' => 'required',
            'professional_details' => 'required',
            'employment_status' => 'filled',
            'total_ratings' => 'filled',
            'avg_ratings' => 'filled',
            'tp_points_balance' => 'filled',
            'verified_status_percentage' => 'required',
            'verified_status' => 'required',
            'fb_url' => 'filled',
            'li_url' => 'filled',
            'tw_url' => 'filled',
            'insta_url' => 'filled',
            'online_cost_per_hour' => 'required|numeric|min:4',
            'offline_cost_per_hour' => 'required|numeric|min:4',
            'institute_cost_per_hour' => 'required|numeric|min:4',
            'tutor_home_cost_per_hour' => 'required|numeric|min:4',
            'student_home_cost_per_hour' => 'required|numeric|min:4',
            'discount_limit' => 'required|numeric|min:1',
            'preferred_board' => 'required',
            'subject_ids' => 'required',
            'topic' => 'required',
            'mode_of_classes' => 'required',
            'request_received' => 'required',
            'request_sent' => 'required',
            'friends_id' => 'required',
            'my_students_ids' => 'required',
            'my_school_id' => 'required',
            'tutor_experience' => 'required|numeric',
            'tutor_commission' => 'required|numeric',
            'notifications' => 'required|min:2|Max:6',
            'parent_ids' => 'required',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {

            $userProfileUpdate = UserDetails::where('user_id', $request->user_id ?? auth()->user()->id)->update($request->all());
            if ($userProfileUpdate) {
                $this->setResponse(false, 'User data updated successfully.');
                return response()->json($this->_response);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    // /* Selected User Profile View */
    public function profile()
    {
        // $validator = Validator::make($request->all(), [
        //     'email' => 'required|exists:users,email',
        // ]);

        // if ($validator->fails()) {
        //     $this->setResponse(true, $validator->errors()->all());
        //     return response()->json($this->_response, 400);
        // }

        try {

            $user = User::where('email', auth()->user()->email)->first();
            return (new UserResource($user))->additional(["error" => false, "message" => "Retrived user profile successfully"]);
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    // /* Selected User Profile View */
    public function userWiseProfile(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|exists:users,email',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {

            $user = User::where('email', $request->email)->first();

            // Fetch only the latest 5 complaints and services
            $user->complaintsAssigned = $user->complaintsAssigned()->latest()->limit(5)->get();
            $user->amcServicesAssigned = $user->amcServicesAssigned()->latest()->limit(5)->get();
        
            if ($user->getRole() == 'customer') {
                return (new CustomerProfileResource($user))->additional(["error" => false, "message" => "Retrived user profile successfully"]);
            } elseif ($user->getRole() == 'employee') {
                return (new EmployeeProfileResource($user))->additional(["error" => false, "message" => "Retrived user profile successfully"]);
            } elseif ($user->getRole() == 'supplier') {
                return (new SupplierProfileResource($user))->additional(["error" => false, "message" => "Retrived user profile successfully"]);
            } else {
                return (new UserResource($user))->additional(["error" => false, "message" => "Retrived user profile successfully"]);
            }


            $user = User::where('email', auth()->user()->email)->first();

            return (new UserResource($user))->additional(["error" => false, "message" => "Retrived user profile successfully"]);
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    public function impersonate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            // 'email' => 'required|exists:users,email',
            'id' => 'required|exists:users,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            // $token = auth()->user()->actingAs($request->email);
            $user = User::find($request->id);
            // $token = Auth::login($user);
            if (!$user->hasRole('admin')) {
                /* Logout user and then imporsonate using anoher use 
                                            OR
                Logout imporsonated user and login with actual user */

                Auth::logout();

                $token = Auth::login($user);
                return $this->respondWithToken($token);
            } else {
                $this->setResponse(false, 'You cant impersonate using admin user.', '401');
                return response()->json($this->_response, 200);
            }
            // $this->_response['data'] = [
            //     "auth_token" => $token,
            // ];

            // return response()->json($this->_response, 200);
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Get the token array structure.
     *
     * @param string $token
     *
     * @return \Illuminate\Http\JsonResponse
     */
    protected function respondWithToken($token)
    {
        return response()->json([
            'data' => (new UserResource(auth()->user())),
            'access_token' => $token,
            'permissions' => PermissionResource::collection(auth()->user()->getAllPermissions()),
            'token_type' => 'bearer',
            'error' => false,
            'expires_in' => auth()->factory()->getTTL() * 60,
            'message' => "Logged in successfully."
        ]);
    }

    public function assignrole(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|exists:users,email',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $user = User::where('email', $request->email)->first();
            $user->assignRole($request->role);
            return response()->json("true", 200);
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*User List By Role */
    public function dropdownByRole(Request $request)
    {
        $role = $request->input('role');
        $loggedInUserId = auth()->id();
        $validator = Validator::make($request->all(), [
            'role' => 'required|in:customer,employee,supplier,admin,all',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => true, 'message' => $validator->errors()->all()], 400);
        }

        try {
            $query = User::query();
            if ($role === 'all') {
                $query->role(['employee', 'customer', 'supplier', 'admin']);

            } else {
                $query->role($role);
            }
            // Exclude the currently logged-in user
            $query->where('id', '<>', $loggedInUserId);

            $userData = $query->orderBy('created_at', 'desc')->get();
            return UserpluckResource::collection($userData)->additional(["error" => false, "message" => '']);
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Verify User Email */
    public function VerifyEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'verify_token' => 'required|exists:users,verify_token',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $user =  User::where('verify_token', $request->verify_token)->first();

            $verifyEmail =  User::where('verify_token', $request->verify_token)->update(['is_verified' => 1, 'verify_token' => '']);

            if ($verifyEmail) {
                // Update profile mail
                $mailData = [
                    'host' => getHost(),
                    'email_subject' => 'Update your profile',
                    'email_template' => 'UpdateProfile',
                ];

                Mail::to($user->email)->queue(new Notification($mailData));

                $this->_response['data'] = '';
                $this->setResponse(false, 'Email verified Successfully.');
                return response()->json($this->_response);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Verify User Directly - for developer use purpose */
    public function directVerify(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|exists:users,email',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $verifyEmail =  User::where('email', $request->email)->update(['is_verified' => 1, 'verify_token' => '']);

            if ($verifyEmail) {
                $this->_response['data'] = '';
                $this->setResponse(false, 'User verified Successfully.');
                return response()->json($this->_response);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /* Request for access */
    public function requestForAccess(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'description' => 'required'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $mailData = [
                'user_name' => auth()->user()->full_name,
                'user_email' => auth()->user()->email,
                'user_tpid' => (auth()->user()->details->tp_id) ? auth()->user()->details->tp_id : 'NA',
                'user_role' => auth()->user()->getRole(),
                'description' => $request->description,
                'email_subject' => 'Request For Access - ' . auth()->user()->full_name,
                'email_template' => 'AccessRequest',
                'host' => getHost()
            ];

            $adminEmails = User::role('admin')->pluck('email')->toArray();

            foreach ($adminEmails as $email) {
                Mail::to($email)->queue(new Notification($mailData));
            }

            $this->setResponse(false, 'Request sent successfully.');
            return response()->json($this->_response);
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /* Test Mail */
    public function testMail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $mailData = [
                'description' => 'This is test mail from tutorpark',
                'email_subject' => 'Test Mail - TutorPark',
                'email_template' => 'Test',
            ];

            Mail::to($request->email)->queue(new Notification($mailData));

            $this->setResponse(false, 'Request sent successfully.');
            return response()->json($this->_response);
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

}
