<?php

namespace App\Http\Controllers;

use App\Http\Resources\StockResource;
use Illuminate\Support\Str;
use App\Models\Task;
use App\Models\Tax;
use App\Models\User;
use App\Models\ActivityLog;
use Illuminate\Http\Request;
use App\Http\Resources\TaxResource;
use Illuminate\Support\Facades\Validator;

class TaxController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('tax_view')) {

                $tax = Tax::Orderby('created_at', 'desc')->get();
                return TaxResource::collection($tax)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'label' => 'required',
            'percentage' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('tax_add')) {

                $tax = Tax::create($request->all()); 
                
                $sourceName = auth()->id();
                $users = User::find($sourceName);
                $sourceId = $users ? $users->first_name : 'Unknown User';
                $data = [
                'updated_by' => $sourceId,
                'activity' => Tax::TAX_CREATE_ACTIVITY,
                'activity_slug' =>Str::of(Tax::TAX_CREATE_ACTIVITY)->slug('_'),
                'target_name' => $tax->label,
                'details' => "Tax '{$tax->label}' created",
            ];
            ActivityLog::log($data);
                $this->_response['data'] = '';
                $this->setResponse(false, 'Tax created successfully.');
                return response()->json($this->_response);

            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:taxes,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('tax_view')) {

                $tax = Tax::find($request->id);

                return (new TaxResource($tax))->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:taxes,id',
            'label' => 'required',
            'percentage' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('tax_edit')) {

                $tax = Tax::find($request->id);

                $tax->update($request->all());

                $sourceName = auth()->id();
                $users = User::find($sourceName);
                $sourceId = $users ? $users->first_name : 'Unknown User';
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => Tax::TAX_UPDATE_ACTIVITY,
                    'activity_slug' =>Str::of(Tax::TAX_UPDATE_ACTIVITY)->slug('_'),
                    'target_name' => $tax->label,
                    'details' => "Tax '{$tax->label}' updated",
                ];
                ActivityLog::log($data);
                $this->_response['data'] = '';
                $this->setResponse(false, 'Tax updated successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:taxes,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('tax_delete')) {

                $tax =  Tax::find($request->id);
                if ($tax) {
                    $tax->delete();

                    $sourceName = auth()->id();
                    $users = User::find($sourceName);
                    $sourceId = $users ? $users->first_name : 'Unknown User';
                $data = [
                'updated_by' => $sourceId,
                'activity' => Tax::TAX_DELETE_ACTIVITY,
                'activity_slug' =>Str::of(Tax::TAX_DELETE_ACTIVITY)->slug('_'),
                'target_name' => $tax->label,
                'details' => "Tax '{$tax->label}' deleted",
            ];
            ActivityLog::log($data);
                    $this->setResponse(false, 'Tax deleted successfully.');
                    return response()->json($this->_response);
                }
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete multiple AMC Data*/
    public function deleteMulti(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'tax_ids' => 'required|array',
            'tax_ids.*' => 'required|exists:taxes,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('tax_delete')) {

                foreach ($request->tax_ids as $id) {
                    $tax =  Tax::find($id);
                    if ($tax) {
                        $tax->delete();
                    }
                }
                $sourceName = auth()->id();
                $users = User::find($sourceName);
                $sourceId = $users ? $users->first_name : 'Unknown User';
            $data = [
            'updated_by' => $sourceId,
            'activity' => Tax::TAX_DELETE_ACTIVITY,
            'activity_slug' =>Str::of(Tax::TAX_DELETE_ACTIVITY)->slug('_'),
            'target_name' => $tax->label,
            'details' => "Tax '{$tax->label}' deleted",
        ];
        ActivityLog::log($data);
                $this->setResponse(false, 'Tax records deleted successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }
}
