<?php

namespace App\Http\Controllers;

use App\Events\SendFcmNotification;
use App\Models\Task;
use App\Models\ActivityLog;
use App\Models\User;
use App\Models\Notifications;
use Illuminate\Http\Request;
use App\Http\Resources\TaskResource;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;

class TaskController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('task_view')) {
                if (auth()->user()->hasPermissionTo('task_own_data')) {
                    if (auth()->user()->getRole() == 'customer') {
                        $tasks = Task::where(['customer_id' => auth()->user()->id])->Orderby('created_at', 'desc')->get();
                    } elseif (auth()->user()->getRole() == 'employee') {
                        $tasks = Task::where(['assign_to_id' => auth()->user()->id])->Orderby('created_at', 'desc')->get();
                    }
                } else {
                    $tasks = Task::Orderby('created_at', 'desc')->get();
                }

                return TaskResource::collection($tasks)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:users,id',
            'date' => 'required|date',
            'assign_date' => 'required|date',
            'status_id' => 'required|exists:categories,id',
            'assign_to_id' => 'required|exists:users,id',
            'task_type_id' => 'required|exists:categories,id',
            'description' => 'required',
            'attachment' => 'filled|mimes:jpg,bmp,png,jpeg,svg,doc,docx,pdf,csv,txt,mp4,x-flv,x-mpegURL,MP2T,3gpp,quicktime,x-msvideo,x-ms-wmv,ogx,oga,ogv,ogg,webm|max:2048',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('task_add')) {

                $task = Task::create($request->except('attachment'));

                if ($request->has('attachment')) {

                    $attachment = $this->uploadFile($request->attachment, 'images/task');
                    if ($attachment != false) {
                        $task->attachment = $attachment;
                    }
                }
                $task->save();

                $sourceName = auth()->id();
                $users = User::find($sourceName);
                $sourceId = $users ? $users->first_name : 'Unknown User';
            $data = [
                'updated_by' => $sourceId,
                'activity' => Task::TASK_CREATE_ACTIVITY,
                'activity_slug' =>Str::of(Task::TASK_CREATE_ACTIVITY)->slug('_'),
                'target_name' => $task->description,
                'details' => "Task for user '{$task->customer->full_name}' created",
            ];
            ActivityLog::log($data);
            $admin = User::role(['admin'])->get();
            $notifications = [
                [
                    'user_id' =>  $admin->first()->id , // Notify the customer
                    'target_id' => $task->id,
                    'updated_by' => $sourceId,
                    'content' => "New {$task->taskType->value} task assigned to {$task->employee->first_name}.",
                    'is_read' => 0,
                    'path' => Task::TASK_PATH
                ],
                [
                    'user_id' => $task->employee->id, // Notify the assigned user
                    'target_id' => $task->id,
                    'updated_by' => $sourceId,
                    'content' => "New {$task->taskType->value} task is assigned to you.",
                    'is_read' => 0,
                    'path' => Task::TASK_PATH
                ],
            ];

            // Send the notifications
            Notifications::send($notifications);
            $notificationData = [
                [
                    'deviceToken' => $task->first()->device_token,
                    'title' => "New task assigned",
                    'body' => "New {$task->taskType->value} task is assigned to {$task->employee->first_name}.",
                ],
                [
                    'deviceToken' => $task->employee->device_token,
                    'title' => "New task assigned",
                    'body' => "New {$task->taskType->value} task is assigned to you.",
                ]
                ];
                event(new SendFcmNotification($notificationData));
                $this->_response['data'] = '';
                $this->setResponse(false, 'Task created successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Task  $task
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:tasks,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('task_view')) {

                $task = Task::find($request->id);

                return (new TaskResource($task))->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Task  $task
     * @return \Illuminate\Http\Response
     */
    public function edit(Task $task)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Task  $task
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Task $task)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:tasks,id',
            'customer_id' => 'required|exists:users,id',
            'date' => 'required|date',
            'assign_date' => 'required|date',
            'status_id' => 'required|exists:categories,id',
            'assign_to_id' => 'required|exists:users,id',
            'task_type_id' => 'required|exists:categories,id',
            'description' => 'required',
            'attachment' => 'filled|mimes:jpg,bmp,png,jpeg,svg,doc,docx,pdf,csv,txt,mp4,x-flv,x-mpegURL,MP2T,3gpp,quicktime,x-msvideo,x-ms-wmv,ogx,oga,ogv,ogg,webm|max:2048',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('task_edit')) {

                $task = Task::find($request->id);
                // if($request->status_id != $request->employee_status_id && auth()->user()->getRole() != 'admin')
                // {
                //     $request->request->add(['is_employee_status_approved' => false]);
                // }else{
                //     $request->request->add(['employee_status_id' => $request->status_id, 'is_employee_status_approved' => true]);
                // }
                $old_status_id = $task->status_id;
                $task->update($request->except(['attachment']));

                if ($request->has('attachment')) {

                    $attachment = $this->uploadFile($request->attachment, 'images/task');
                    if ($attachment != false) {
                        $task->attachment = $attachment;
                    }
                }
                $task->save();
                $sourceName = auth()->id();
                $users = User::find($sourceName);
                $sourceId = $users ? $users->first_name : 'Unknown User';
                $admin = User::role(['admin'])->get();
                if ($old_status_id != $request->status_id) {
                    switch ($request->status_id) {
                        case 33: // Complaint is in progress
                            $notifications = [
                                // [
                                //     'user_id' => $task->customer->id,
                                //     'target_id' => $task->id,
                                //     'updated_by' => $sourceId,
                                //     'content' => "Your task {$task->task_no} for {$task->taskType->value} has been marked as open.",
                                //     'is_read' => 0,
                                //     'path' => Task::TASK_PATH
                                // ],
                                [
                                'user_id' =>  $admin->first()->id,
                                'target_id' => $task->id,
                                'updated_by' => $sourceId,
                                'content' => "Task {$task->task_no} for {$task->taskType->value} has been marked as open.",
                                'is_read' => 0,
                                'path' => Task::TASK_PATH
                            ],
                            [
                                'user_id' => $task->employee->id,
                                'target_id' => $task->id,
                                'updated_by' => $sourceId,
                                'content' => "Task {$task->task_no} for {$task->taskType->value} has been marked as open.",
                                'is_read' => 0,
                                'path' => Task::TASK_PATH
                            ],];
                            Notifications::send($notifications);
                            $notificationData = [
                                [
                                    'deviceToken' => $admin->first()->device_token,
                                    'title' => "Task has been marked as open.",
                                    'body' => "Task {$task->task_no} for {$task->taskType->value} has been marked as open.",
                                ],
                                [
                                    'deviceToken' => $task->employee->device_token,
                                    'title' => "Task has been marked as open.",
                                    'body' => "Task {$task->task_no} for {$task->taskType->value} has been marked as open.",
                                ]
                            ];
                            event(new SendFcmNotification($notificationData));
                            break;
                        case 34: // Complaint is in progress
                            $notifications = [
                                // [
                            //     'user_id' => $task->customer->id,
                            //     'target_id' => $task->id,
                            //     'updated_by' => $sourceId,
                            //     'content' => "Your task {$task->task_no} for {$task->taskType->value} has been marked as in progress.",
                            //     'is_read' => 0,
                            //     'path' => Task::TASK_PATH
                            // ],
                            [
                                'user_id' =>  $admin->first()->id,
                                'target_id' => $task->id,
                                'updated_by' => $sourceId,
                                'content' => "Task {$task->task_no} for {$task->taskType->value} has been marked as in progress.",
                                'is_read' => 0,
                                'path' => Task::TASK_PATH
                            ],
                            [
                                'user_id' => $task->employee->id,
                                'target_id' => $task->id,
                                'updated_by' => $sourceId,
                                'content' => "Task {$task->task_no} for {$task->taskType->value} has been marked as in progress.",
                                'is_read' => 0,
                                'path' => Task::TASK_PATH
                            ],];
                            Notifications::send($notifications);
                            $notificationData = [
                                [
                                    'deviceToken' => $admin->first()->device_token,
                                    'title' => "Task has been marked as in progress.",
                                    'body' => "Task {$task->task_no} for {$task->taskType->value} has been marked as in progress.",
                                ],
                                [
                                    'deviceToken' => $task->employee->device_token,
                                    'title' => "Task has been marked as in progress.",
                                    'body' => "Task {$task->task_no} for {$task->taskType->value} has been marked as in progress.",
                                ]
                            ];
                            event(new SendFcmNotification($notificationData));
                            break;
                        case 35: // Complaint is closed and solved
                            $notifications = [
                                // / [
                            //     'user_id' => $task->customer->id,
                            //     'target_id' => $task->id,
                            //     'updated_by' => $sourceId,
                            //     'content' => "Your task {$task->task_no} for {$task->taskType->value} has been closed.",
                            //     'is_read' => 0,
                            //     'path' => Task::TASK_PATH
                            // ],
                            [
                                'user_id' => $task->employee->id,
                                'target_id' => $task->id,
                                'updated_by' => $sourceId,
                                'content' => "Task {$task->task_no} for {$task->taskType->value} has been marked as completed.",
                                'is_read' => 0,
                                'path' => Task::TASK_PATH
                            ],
                            [
                                'user_id' => $admin->first()->id,
                                'target_id' => $task->id,
                                'updated_by' => $sourceId,
                                'content' => "Task {$task->task_no} for {$task->taskType->value} has been marked as completed.",
                                'is_read' => 0,
                                'path' => Task::TASK_PATH
                            ] ];
                            Notifications::send($notifications);
                            $notificationData = [
                                [
                                    'deviceToken' => $task->employee->device_token,
                                    'title' => "Task has been marked as completed.",
                                    'body' => "Task {$task->task_no} for {$task->taskType->value} has been marked as completed.",
                                ],
                                [
                                    'deviceToken' => $admin->first()->device_token,
                                    'title' => "Task has been marked as completed.",
                                    'body' => "Task {$task->task_no} for {$task->taskType->value} has been marked as completed.",
                                ]
                            ];
                            event(new SendFcmNotification($notificationData));
                            break;
                        // Add more cases for other status changes if needed
                        default:
                            // No action for other status changes
                            break;
                    }};
            $data = [
                'updated_by' => $sourceId,
                'activity' => Task::TASK_UPDATE_ACTIVITY,
                'activity_slug' =>Str::of(Task::TASK_UPDATE_ACTIVITY)->slug('_'),
                'target_name' => $task->description,
                'details' => "Task for user '{$task->customer->full_name}' updated",
            ];
            ActivityLog::log($data);
                $this->_response['data'] = '';
                $this->setResponse(false, 'Task updated successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:tasks,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('task_delete')) {

                $task =  Task::find($request->id);
                if ($task) {
                    $task->delete();

                    $sourceName = auth()->id();
                    $users = User::find($sourceName);
                    $sourceId = $users ? $users->first_name : 'Unknown User';
                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Task::TASK_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(Task::TASK_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $task->description,
                        'details' => "Task for user '{$task->customer->full_name}' deleted",
                    ];
            ActivityLog::log($data);
                    $this->setResponse(false, 'Task deleted successfully.');
                    return response()->json($this->_response);
                }
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete multiple AMC Data*/
    public function deleteMultiTask(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'task_ids' => 'required|array',
            'task_ids.*' => 'required|exists:tasks,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('task_delete')) {

                foreach ($request->task_ids as $id) {
                    $task =  Task::find($id);
                    if ($task) {
                        $task->delete();
                    }
                }
                $sourceName = auth()->id();
                $users = User::find($sourceName);
                $sourceId = $users ? $users->first_name : 'Unknown User';
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => Task::TASK_DELETE_ACTIVITY,
                    'activity_slug' =>Str::of(Task::TASK_DELETE_ACTIVITY)->slug('_'),
                    'target_name' => $task->description,
                    'details' => "Task for user '{$task->customer->full_name}' deleted",
                ];
        ActivityLog::log($data);
                $this->setResponse(false, 'Tasks deleted successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function rejectEmployeeStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:tasks,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $task =  Task::find($request->id);
            if ($task) {
                $task->update(['is_employee_status_approved' => true]);

                $this->setResponse(false, 'Status rejected successfully.');
                return response()->json($this->_response);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function approveEmployeeStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:tasks,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $task =  Task::find($request->id);
            if ($task) {
                $task->update(['status_id' => $task->employee_status_id, 'is_employee_status_approved' => true]);

                $this->setResponse(false, 'Status approved successfully.');
                return response()->json($this->_response);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }
}
