<?php

namespace App\Http\Controllers;

use App\Events\SendFcmNotification;
use App\Models\Tax;
use App\Models\Sale;
use App\Models\Setting;
use App\Models\ActivityLog;
use App\Models\User;
use App\Models\Notifications;
use Illuminate\Http\Request;
use App\Mail\SaleInvoice;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use App\Http\Resources\SaleResource;
use Illuminate\Support\Facades\Storage;

class SaleController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('sell_view')) {
                if (auth()->user()->hasPermissionTo('sell_own_data')) {
                    if (auth()->user()->getRole() == 'customer') {
                        $sales = Sale::where(['customer_id' => auth()->user()->id])->Orderby('created_at', 'desc')->get();
                    }
                } else {
                    $sales = Sale::Orderby('created_at', 'desc')->get();
                }

                return SaleResource::collection($sales)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:users,id',
            'date' => 'required|date',
            'status_id' => 'required|exists:categories,id',
            'discount' => 'required|numeric',
            'product_id' => 'required|array',
            'product_id.*' => 'required|exists:products,id',
            'quantity' => 'required|array',
            'quantity.*' => 'required|numeric',
            'rate' => 'required|array',
            'rate.*' => 'required|numeric',
            'amount' => 'required|array',
            'amount.*' => 'required|numeric',
            'taxes' => 'filled|array',
            'taxes.*' => 'required|exists:taxes,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('sell_add')) {

                $sale = Sale::create($request->all());

                $amount = 0;

                // Quotation Selected Products
                foreach ($request->product_id as $key => $value) {
                    $saleProduct = $sale->productDetails()->create([
                        'product_id' => $request->product_id[$key],
                        'quantity' => (int)$request->quantity[$key],
                        'rate' => $request->rate[$key],
                        'amount' => $request->amount[$key],
                    ]);
                    $amount += $request->amount[$key];
                }

                $sale->sub_total = $amount;

                //Discount amount calculation
                $discountAmount = 0;
                if ($request->discount && (int)$request->discount > 0) {
                    $discountAmount = ($request->discount / 100) * $amount;
                }
                $sale->discount_amount = $discountAmount;

                $totalTax = 0;
                // Tax calculation
                foreach ($request->taxes as $tax_id) {
                    $taxRow = Tax::find($tax_id);
                    $taxAmount = ($taxRow->percentage / 100) * ($amount - $discountAmount);
                    $totalTax += $taxAmount;
                    $addedTaxes[] = [
                        "label" => $taxRow->label,
                        "percentage" => $taxRow->percentage,
                        "tax_amount" => $taxAmount,
                    ];
                }


                $sale->tax_amount = $totalTax;
                $sale->gross_total = $amount + $totalTax - $discountAmount;
                $sale->paid_amount = $amount + $totalTax - $discountAmount;
                $sale->due_amount = 0;

                $sale->save();

                $sale->taxes()->attach($request->taxes);

                // Quotation mail to customer

                $settingData = Setting::get();
                $settingData = $settingData->first();
                $systemName = $settingData->system_name;
                // Get email template content using helper
                $emailTemplate = getEmailContent('sales');
                $logo = getSetting('logo');
                $logo_image = url('storage/' . $logo);
                $footer_image = url('storage/email_image/footer.png');
                $support_image = url('storage/email_image/support.png');
                $service_image = url('storage/email_image/service.png');
                $history_image = url('storage/email_image/history.png');

                if ($emailTemplate) {
                    $pdfName = "sale-" . $sale->id . ".pdf";
                    // Pdf::loadView('pdf.Sale', ['data'=>$sale,'setting'=>$settingData])->save(public_path().'/'.$pdfName)->stream($pdfName);

                    $saleInvoice = Pdf::loadView('pdf.Sale', ['data' => $sale, 'setting' => $settingData, 'taxes' => $addedTaxes]);
                    $content = $saleInvoice->download()->getOriginalContent();
                    Storage::disk('public')->put('invoice/' . $pdfName, $content);

                    $mailformat = $emailTemplate->body;
                    $serch = array('{ username }', '{ systemname }');
                    $replace = array($sale->customer->full_name, $systemName);
                    $message_content = str_replace($serch, $replace, $mailformat);

                    // Welcome Mail
                    $mailData = [
                        'host' => getHost(),
                        'subject' => $emailTemplate->subject,
                        'content' => $message_content,
                        'attachementPath' => "invoice/{$pdfName}", //$saleInvoice->output()
                        'logo_image' => $logo_image,
                        'footer_image' => $footer_image,
                        'support_image' => $support_image,
                        'service_image' => $service_image,
                        'history_image' => $history_image
                    ];
                    Mail::to($sale->customer->email)->send(new SaleInvoice($mailData));
                }

                    $sourceName = auth()->id();
                    $users = User::find($sourceName);
                    $sourceId = $users ? $users->first_name : 'Unknown User';
                    $user = auth()->user();
                    $targetName = $sale->customer->full_name;
                    $productname = $sale->productDetails->first()->product->name;
                    $activityDetails = "{$user->first_name} created a sale for {$productname}";
                    switch ($request->status_id) {
                        case 25: // Paid
                            $content = "Your purchase for {$productname} marked as Paid.";
                            $title = "Purchase Paid successfully";
                            break;
                        case 26: // Partial Paid
                            $content = "Your purchase for {$productname} marked as Partial Paid.";
                            $title = "Partial Payment";
                            break;
                        case 27: // Unpaid
                            $content = "Your purchase for {$productname} marked as Unpaid.";
                            $title = "Unpaid Purchase";
                            break;
                        default:
                            $content = "Your purchase for {$productname} is updated.";
                            $title = "Purchase Update";
                            break;
                    }
                    $notifications = [
                        [
                            'user_id' => $sale->customer_id,
                            'target_id' => $sale->id,
                            'updated_by' => $sourceId,
                            'content' => $content,
                            'is_read' => 0,
                            'path' => Sale::SELL_PATH
                        ],
                    ];
                    Notifications::send($notifications);
                    $notificationData = [
                        [
                            'deviceToken' => $sale->customer->device_token,
                            'title' => $title,
                            'body' => $content,
                        ],
                    ];
                    event(new SendFcmNotification($notificationData));

                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Sale::SELL_CREATE_ACTIVITY,
                        'activity_slug' =>Str::of(Sale::SELL_CREATE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);
                $this->_response['data'] = '';
                $this->setResponse(false, 'Sale added successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Sale  $sale
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:sales,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('sell_view')) {

                $sale = Sale::find($request->id);

                return (new SaleResource($sale))->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Quotation  $quotation
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:sales,id',
            'customer_id' => 'required|exists:users,id',
            'date' => 'required|date',
            'status_id' => 'required|exists:categories,id',
            'discount' => 'required|numeric',
            'product_id' => 'required|array',
            'product_id.*' => 'required|exists:products,id',
            'quantity' => 'required|array',
            'quantity.*' => 'required|numeric',
            'rate' => 'required|array',
            'rate.*' => 'required|numeric',
            'amount' => 'required|array',
            'amount.*' => 'required|numeric',
            'taxes' => 'filled|array',
            'taxes.*' => 'required|exists:taxes,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('sell_edit')) {

            $sale = Sale::find($request->id);
            $sale->update($request->all());
            $amount = 0;

            $sale->productDetails()->delete();

            // Quotation Selected Products
            foreach ($request->product_id as $key => $value) {
                $saleProduct = $sale->productDetails()->create([
                    'product_id' => $request->product_id[$key],
                    'quantity' => (int)$request->quantity[$key],
                    'rate' => $request->rate[$key],
                    'amount' => $request->amount[$key],
                ]);
                $amount += $request->amount[$key];
            }

            //Discount amount calculation
            $discountAmount = 0;
            if ($request->discount && (int)$request->discount > 0) {
                $discountAmount = ((int)$request->discount / 100) * $amount;
            }
            $sale->discount_amount = $discountAmount;

            $totalTax = 0;
            $addedTaxes = [];
            // Tax calculation
            foreach ($request->taxes as $tax_id) {
                $taxRow = Tax::find($tax_id);
                $taxAmount =  ($taxRow->percentage / 100) * ($amount - $discountAmount);
                $totalTax += $taxAmount;
                $addedTaxes[] = [
                    "label" => $taxRow->label,
                    "percentage" => $taxRow->percentage,
                    "tax_amount" => $taxAmount,
                ];
            }

            $sale->sub_total = $amount;
            $sale->tax_amount = $totalTax;
            $sale->gross_total = $amount + $totalTax - $discountAmount;
            $sale->save();
            // $quotation->taxes()->detach($quotation->tax_ids);

            $sale->taxes()->sync($request->taxes);
            $settingData = Setting::get();
            $settingData = $settingData->first();

            $pdfName = "sale-" . $sale->id . ".pdf";
            $saleInvoice = Pdf::loadView('pdf.Sale', ['data' => $sale, 'setting' => $settingData, 'taxes' => $addedTaxes]);
            $content = $saleInvoice->download()->getOriginalContent();

            if (Storage::disk('public')->exists("invoice/{$pdfName}")) {
                Storage::disk('public')->delete("invoice/{$pdfName}");
            }

            Storage::disk('public')->put('invoice/' . $pdfName, $content);

                    $sourceName = auth()->id();
                    $users = User::find($sourceName);
                    $sourceId = $users ? $users->first_name : 'Unknown User';
                    $user = auth()->user();
                    $targetName = $sale->customer->full_name;
                    $activityDetails = "{$user->first_name} created a sale for {$targetName}";

                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Sale::SELL_UPDATE_ACTIVITY,
                        'activity_slug' =>Str::of(Sale::SELL_UPDATE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);
                    $productname = $sale->productDetails->first()->product->name;
                    switch ($request->status_id) {
                        case 25: // Paid
                            $content = "Your purchase for {$productname} marked as Paid.";
                            $title = "{$productname} Paid";
                            break;
                        case 26: // Partial Paid
                            $content = "Your purchase for {$productname} marked as Partial Paid.";
                            $title = "{$productname} Partial Paid";
                            break;
                        case 27: // Unpaid
                            $content = "Your purchase for {$productname} marked as Unpaid.";
                            $title = "{$productname} Unpaid";
                            break;
                        default:
                            break;
                    }
                    $notifications = [
                        [
                            'user_id' => $sale->customer_id,
                            'target_id' => $sale->id,
                            'updated_by' => $sourceId,
                            'content' => $content,
                            'is_read' => 0,
                            'path' => Sale::SELL_PATH
                        ],
                    ];
                    Notifications::send($notifications);
                    $notificationData = [
                        [
                            'deviceToken' => $sale->customer->device_token,
                            'title' => $title,
                            'body' => $content,
                        ]
                    ];
                    event(new SendFcmNotification($notificationData));
            $this->_response['data'] = '';
            $this->setResponse(false, 'Sale updated successfully.');
            return response()->json($this->_response);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:sales,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('sell_delete')) {

                $sale =  Sale::find($request->id);
                if ($sale) {
                    $sale->productDetails()->delete();
                    $sale->delete();

                    $sourceName = auth()->id();
                    $users = User::find($sourceName);
                    $sourceId = $users ? $users->first_name : 'Unknown User';
                    $user = auth()->user();
                    $targetName = $sale->customer->full_name; // Assuming this is the target name
                    $activityDetails = "{$user->first_name} deleted a sale for {$targetName}";

                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Sale::SELL_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(Sale::SELL_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);
                    $this->setResponse(false, 'Sale deleted successfully.');
                    return response()->json($this->_response);
                }
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete multiple AMC Data*/
    public function deleteMultiQuotation(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'sale_ids' => 'required|array',
            'sale_ids.*' => 'required|exists:sales,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('sell_delete')) {

                foreach ($request->sale_ids as $id) {
                    $sale =  Sale::find($id);
                    if ($sale) {
                        $sale->productDetails()->delete();
                        $sale->delete();
                    }
                }

                $sourceName = auth()->id();
                $users = User::find($sourceName);
                $sourceId = $users ? $users->first_name : 'Unknown User';
                    $user = auth()->user();
                    $targetName = $sale->customer->full_name; // Assuming this is the target name
                    $activityDetails = "{$user->first_name} deleted a sale for {$targetName}";

                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Sale::SELL_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(Sale::SELL_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);
                $this->setResponse(false, 'Sales deleted successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    public function addPayment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'sale_id' => 'required|exists:sales,id',
            'amount' => 'required|numeric|min:0',
            'payment_date' => 'required|date',
            'payment_method' => 'required|string',
            'notes' => 'nullable|string',
        ]);
    
        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
    
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('sell_edit')) {
                $sale = Sale::findOrFail($request->sale_id);
                
                $payment = $sale->paymentHistories()->create([
                    'amount' => $request->amount,
                    'payment_date' => $request->payment_date,
                    'payment_method' => $request->payment_method,
                    'notes' => $request->notes,
                ]);
    
                $sale->updatePaymentStatus();
    
                // Regenerate PDF
                $settingData = Setting::first();
                $pdfName = "sale-" . $sale->id . ".pdf";
    
                $addedTaxes = [];
                foreach ($sale->taxes as $tax) {
                    $taxAmount = ($tax->percentage / 100) * ($sale->sub_total - $sale->discount_amount);
                    $addedTaxes[] = [
                        "label" => $tax->label,
                        "percentage" => $tax->percentage,
                        "tax_amount" => $taxAmount,
                    ];
                }
    
                $saleInvoice = Pdf::loadView('pdf.Sale', [
                    'data' => $sale, 
                    'setting' => $settingData, 
                    'taxes' => $addedTaxes
                ]);
                $content = $saleInvoice->download()->getOriginalContent();
    
                if (Storage::disk('public')->exists("invoice/{$pdfName}")) {
                    Storage::disk('public')->delete("invoice/{$pdfName}");
                }
    
                Storage::disk('public')->put('invoice/' . $pdfName, $content);
    
                // Add notification and activity log
                $sourceName = auth()->id();
                $user = User::find($sourceName);
                $sourceId = $user ? $user->first_name : 'Unknown User';
                $targetName = $sale->customer->full_name;
                $activityDetails = "{$user->first_name} added a payment of {$request->amount} for sale #{$sale->id}";
    
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => 'Payment Added',
                    'activity_slug' => 'payment_added',
                    'target_name' => $targetName,
                    'details' => $activityDetails,
                ];
                ActivityLog::log($data);
    
                $notificationContent = "A payment of {$request->amount} has been added to your purchase (Sale #{$sale->id}).";
                $notifications = [
                    [
                        'user_id' => $sale->customer_id,
                        'target_id' => $sale->id,
                        'updated_by' => $sourceId,
                        'content' => $notificationContent,
                        'is_read' => 0,
                        'path' => Sale::SELL_PATH
                    ],
                ];
                Notifications::send($notifications);
    
                $notificationData = [
                    [
                        'deviceToken' => $sale->customer->device_token,
                        'title' => 'Payment Added',
                        'body' => $notificationContent,
                    ]
                ];
                event(new SendFcmNotification($notificationData));
    
                $this->_response['data'] = new SaleResource($sale);
                $this->setResponse(false, 'Payment added successfully and invoice updated.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }
}
