<?php

namespace App\Http\Controllers;

use DB;
use Carbon\Carbon;
use App\Models\Sale;
use App\Models\Complaint;
use App\Models\AmcService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ReportController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function complaintReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'from_date' => 'required|date',
            'to_date' => 'required|date',
            'status_id' => 'filled|exists:categories,id',
            'customer_id' => 'filled|exists:users,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {

            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('report_view')) {

                $fromDate = date("Y-m-d", strtotime($request->from_date));
                $toDate = date("Y-m-d", strtotime($request->to_date));
                $statusId = $request->status_id;
                $customerId = $request->customer_id;

                $report = Complaint::when($fromDate, function ($query, $fromDate) {
                    return $query->whereDate('assign_date', '>=', $fromDate);
                })
                    ->when($toDate, function ($query, $toDate) {
                        return $query->whereDate('assign_date', '<=', $toDate);
                    })
                    ->when($statusId, function ($query, $statusId) {
                        return $query->where(['status_id' => $statusId]);
                    })
                    ->when($customerId, function ($query, $customerId) {
                        return $query->where('customer_id', '=', $customerId);
                    })
                    ->Orderby('created_at', 'desc')
                    ->get()
                    ->groupBy(function ($item) {
                        return Carbon::parse($item->assign_date)->format('m');
                    });

                $data['labels'] = [];
                $data['values'] = [];

                foreach ($report as $key => $value) {
                    $montName = date('M-Y', mktime(0, 0, 0, $key, 10));

                    $data['labels'][] = $montName;
                    $data['values'][] = count($value);
                }

                return response()->json([
                    'data' => $data,
                    'error' => false,
                    'message' => ''
                ], 200);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function saleReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'from_date' => 'required|date',
            'to_date' => 'required|date',
            'customer_id' => 'filled|exists:users,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {

            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('report_view')) {

                $fromDate = date("Y-m-d", strtotime($request->from_date));
                $toDate = date("Y-m-d", strtotime($request->to_date));
                $customerId = $request->customer_id;

                // $report = Sale::when($fromDate, function ($query, $fromDate) {
                //     return $query->whereDate('date', '>=', $fromDate);
                // })
                //     ->when($toDate, function ($query, $toDate) {
                //         return $query->whereDate('date', '<=', $toDate);
                //     })
                //     ->when($customerId, function ($query, $customerId) {
                //         return $query->where('customer_id', '=', $customerId);
                //     })
                //     ->Orderby('created_at', 'desc')
                //     ->get()
                //     ->groupBy(function ($item) {
                //         return Carbon::parse($item->date)->format('m');
                //     });
                $report = Sale::whereDate('date', '>=', $fromDate)
                    ->whereDate('date', '<=', $toDate)
                    ->when($customerId, function ($query, $customerId) {
                        return $query->where('customer_id', '=', $customerId);
                    })
                    ->Orderby('created_at', 'desc')
                    ->get()
                    ->groupBy(function ($item) {
                        return Carbon::parse($item->date)->format('m');
                    });

                $data['labels'] = [];
                $data['values'] = [];

                foreach ($report as $key => $value) {
                    $montName = date('M-Y', mktime(0, 0, 0, $key, 10));

                    $data['labels'][] = $montName;
                    $data['values'][] = count($value);
                }

                return response()->json([
                    'data' => $data,
                    'error' => false,
                    'message' => ''
                ], 200);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function serviceReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'from_date' => 'required|date',
            'to_date' => 'required|date',
            'customer_id' => 'filled|exists:users,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {

            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('report_view')) {

                $fromDate = date("Y-m-d", strtotime($request->from_date));
                $toDate = date("Y-m-d", strtotime($request->to_date));
                $customerId = $request->customer_id;

                $report = AmcService::when($fromDate, function ($query, $fromDate) {
                    return $query->whereDate('service_date', '>=', $fromDate);
                })
                    ->when($toDate, function ($query, $toDate) {
                        return $query->whereDate('service_date', '<=', $toDate);
                    })
                    ->when($customerId, function ($query, $customerId) {
                        return $query->where('customer_id', '=', $customerId);
                    })
                    ->Orderby('created_at', 'desc')
                    ->get()
                    ->groupBy(function ($item) {
                        return Carbon::parse($item->service_date)->format('m');
                    });

                $data['labels'] = [];
                $data['values'] = [];

                foreach ($report as $key => $value) {
                    $montName = date('M-Y', mktime(0, 0, 0, $key, 10));

                    $data['labels'][] = $montName;
                    $data['values'][] = count($value);
                }

                return response()->json([
                    'data' => $data,
                    'error' => false,
                    'message' => ''
                ], 200);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }
}
