<?php

namespace App\Http\Controllers;

use App\Events\SendFcmNotification;
use DB;
use Illuminate\Http\Request;
use App\Models\ActivityLog;
use App\Models\Product;
use App\Models\User;
use App\Models\Notifications;
use App\Models\PurchaseProducts;
use App\Http\Resources\PurchaseResource;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use App\Models\Purchase;

class PurchaseController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('purchase_view')) {

                $purchase = Purchase::Orderby('created_at', 'desc')->get();
                return PurchaseResource::collection($purchase)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'date' => 'required|date',
            'status' => 'required|exists:categories,id',
            'contact_person' => 'required',
            'supplier_id' => 'required|exists:users,id',
            'product_id' => 'required|array',
            'product_id.*' => 'required|exists:products,id',
            'quantity' => 'required|array',
            'quantity.*' => 'required|numeric',
            'rate' => 'required|array',
            'rate.*' => 'required|numeric',
            'amount' => 'required|array',
            'amount.*' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('purchase_add')) {

                $purchase = Purchase::create($request->all());

                // Purchase Selected Products
                foreach ($request->product_id as $key => $value) {
                    $purchaseProduct = $purchase->productDetails()->create([
                        'product_id' => $request->product_id[$key],
                        'quantity' => (int)$request->quantity[$key],
                        'rate' => $request->rate[$key],
                        'amount' => $request->amount[$key],
                    ]);

                    if ($purchaseProduct->stock()->count()) {
                        $purchaseProduct->stock()->increment('stock', (int)$request->quantity[$key]);
                    } else {
                        $purchaseProduct->stock()->create(['stock' => (int)$request->quantity[$key]]);
                    }
                }

                $sourceName = auth()->id();
                $users = User::find($sourceName);
                $sourceId = $users ? $users->first_name : 'Unknown User';
            $activityDetails = "Purchase for product '";
                $product = Product::find($value);
                $activityDetails .= "{$product->name}' created";

            $data = [
                'updated_by' => $sourceId,
                'activity' => Purchase::PURCHASE_CREATE_ACTIVITY,
                'activity_slug' =>Str::of(Purchase::PURCHASE_CREATE_ACTIVITY)->slug('_'),
                'target_name' => $product->name,
                'details' => $activityDetails,
            ];
            ActivityLog::log($data);
            $admin = User::role(['admin'])->get();
            if(auth()->user()->getRole() == 'admin') {
            $notifications = [
                [
                    'user_id' => $purchase->supplier_id,
                    'target_id' => $purchase->id,
                    'updated_by' => $sourceId,
                    'content' => "New purchase order placed for {$product->name}. Check details now.",
                    'is_read' => 0, // Set as unread (0) by default
                    'path' => Purchase::PURCHASE_PATH,
                ],
            ];
            
            // Send multiple notifications
            Notifications::send($notifications);
            $notificationData = [
                [
                'deviceToken' =>  $purchase->supplier->device_token,
                'title' => 'New Purchase Order',
                'body' => "A new purchase order has been placed for {$product->name}.",
                ]
            ];
            event(new SendFcmNotification($notificationData));
        }
        else{
            $notifications = [
                [
                    'user_id' => $purchase->supplier_id,
                    'target_id' => $purchase->id,
                    'updated_by' => $sourceId,
                    'content' => "New purchase order placed for {$product->name}. Check details now.",
                    'is_read' => 0, // Set as unread (0) by default
                    'path' => Purchase::PURCHASE_PATH,
                ],
                [
                    'user_id' =>$admin->first()->id,
                    'target_id' => $purchase->id,
                    'updated_by' => $sourceId,
                    'content' => "A new purchase order has been placed for {$product->name}.",
                    'is_read' => 0, // Set as unread (0) by default
                    'path' => Purchase::PURCHASE_PATH,
                ],
            ];
            
            // Send multiple notifications
            Notifications::send($notifications);
            $notificationData = [
                [
                'deviceToken' =>  $purchase->supplier->device_token,
                'title' => 'New Purchase Order',
                'body' => "A new purchase order has been placed for {$product->name}.",
                ],
                [
                'deviceToken' =>  $admin->first()->device_token,
                'title' => 'New Purchase Order',
                'body' => "A new purchase order has been placed for {$product->name}.",
                ]
            ];
            event(new SendFcmNotification($notificationData));
        }
                $this->_response['data'] = '';
                $this->setResponse(false, 'Purchase created successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:purchases,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('purchase_view')) {

                $purchase = Purchase::find($request->id);

                return (new PurchaseResource($purchase))->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:purchases,id',
            'date' => 'required|date',
            'status' => 'required|exists:categories,id',
            'contact_person' => 'required',
            'supplier_id' => 'required|exists:users,id',
            'product_id' => 'required|array',
            'product_id.*' => 'required|exists:products,id',
            'quantity' => 'required|array',
            'quantity.*' => 'required|numeric',
            'rate' => 'required|array',
            'rate.*' => 'required|numeric',
            'amount' => 'required|array',
            'amount.*' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        // try {
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('purchase_edit')) {

            $purchase = Purchase::find($request->id);
            $old_status_id = $purchase->status;
            $purchase->update($request->all());
            // Delete old records
            // $purchase->productDetails()->delete();

            // Purchase Selected Products
            foreach ($request->product_id as $key => $value) {
                $oldRecord = $purchase->productDetails()->where(['product_id' => $request->product_id[$key]])->first();
                $difference = ($oldRecord && $oldRecord->quantity) ? $request->quantity[$key] - $oldRecord->quantity : 0;

                // Delete old records
                $purchase->productDetails()->where(['product_id' => $request->product_id[$key]])->delete();

                $purchaseProduct = $purchase->productDetails()->create([
                    'product_id' => $request->product_id[$key],
                    'quantity' => (int)$request->quantity[$key],
                    'rate' => $request->rate[$key],
                    'amount' => $request->amount[$key],
                ]);

                if ($purchaseProduct->stock()->count()) {
                    $purchaseProduct->stock()->increment('stock', $difference);
                } else {
                    $purchaseProduct->stock()->create(['stock' => (int)$request->quantity[$key]]);
                }

                $sourceName = auth()->id();
                $users = User::find($sourceName);
                $sourceId = $users ? $users->first_name : 'Unknown User';
                $targetProduct = Product::find($value);
                $targetName = $targetProduct ? $targetProduct->name : 'Unknown Product';

                $activityDetails = "Purchase for product '{$targetName}' updated";

                $data = [
                    'updated_by' => $sourceId,
                    'activity' => Purchase::PURCHASE_UPDATE_ACTIVITY,
                    'activity_slug' =>Str::of(Purchase::PURCHASE_UPDATE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => $activityDetails,
                ];
                ActivityLog::log($data);
            }
                $admin = User::role(['admin'])->get();
                if ($old_status_id != $request->status) {
                    if ($request->status == 21) {
                        $notifications = [[
                            'user_id' => $purchase->supplier_id,
                            'target_id' => $purchase->id,
                            'updated_by' => $sourceId,
                            'content' => "Purchase order for {$targetName} has been marked as delivered.",
                            'is_read' => 0, // Set as unread (0) by default
                            'path' => Purchase::PURCHASE_PATH,
                        ],
                        [
                            'user_id' => $admin->first()->id,
                            'target_id' => $purchase->id,
                            'updated_by' => $sourceId,
                            'content' => "A purchase order for {$targetName} has been marked as delivered.",
                            'is_read' => 0, // Set as unread (0) by default
                            'path' => Purchase::PURCHASE_PATH,
                        ]];
                            Notifications::send($notifications);
                            $notificationData = [
                                [
                                    'deviceToken' => $purchase->supplier->device_token,
                                    'title' => "Purchase Order Delivered",
                                    'body' => "A purchase order for {$targetName} has been marked as delivered.",
                                ],
                                [
                                    'deviceToken' => $admin->first()->device_token,
                                    'title' => "Purchase Order Delivered",
                                    'body' => "A purchase order for {$targetName} has been marked as delivered.",
                                ]
                                ];
                                event(new SendFcmNotification($notificationData));
                    }}
            $this->_response['data'] = '';
            $this->setResponse(false, 'Purchase updated successfully.');
            return response()->json($this->_response);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
        // } catch (\Exception $e) {
        //     $this->setResponse(true,  $e->getMessage());
        //     return response()->json($this->_response, 500);
        // }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:purchases,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('purchase_delete')) {

                $purchase =  Purchase::find($request->id);
                if ($purchase) {
                    $purchaseProducts = $purchase->productDetails;

                    foreach ($purchaseProducts as $purchaseProduct) {
                        $product = Product::find($purchaseProduct->product_id);
                        $targetName = $product ? $product->name : 'Unknown Product';

                        // Log activity for each deleted product
                        $sourceName = auth()->id();
                        $users = User::find($sourceName);
                        $sourceId = $users ? $users->first_name : 'Unknown User';
                        $activityDetails = "Purchase for product '{$targetName}' deleted";

                        $data = [
                            'updated_by' => $sourceId,
                            'activity' => Purchase::PURCHASE_DELETE_ACTIVITY,
                            'activity_slug' =>Str::of(Purchase::PURCHASE_DELETE_ACTIVITY)->slug('_'),
                            'target_name' => $targetName,
                            'details' => $activityDetails,
                        ];
                        ActivityLog::log($data);
                    }

                    // Now delete the purchase and its products
                    $purchase->productDetails()->delete();
                    $purchase->delete();

                    $this->setResponse(false, 'Purchase record deleted successfully.');
                    return response()->json($this->_response);
                }
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete multiple AMC Data*/
    public function deleteMulti(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'purchase_ids' => 'required|array',
            'purchase_ids.*' => 'required|exists:purchases,id',
        ]); 

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('purchase_delete')) {

                foreach ($request->purchase_ids as $id) {
                    $purchase = Purchase::find($id);
                    if ($purchase) {
                        $purchaseProducts = $purchase->productDetails;

                        foreach ($purchaseProducts as $purchaseProduct) {
                            $product = Product::find($purchaseProduct->product_id);
                            $targetName = $product ? $product->name : 'Unknown Product';

                            // Log activity for each deleted product
                            $sourceName = auth()->id();
                            $users = User::find($sourceName);
                            $sourceId = $users ? $users->first_name : 'Unknown User';
                            $activityDetails = "Purchase for product '{$targetName}' deleted";

                            $data = [
                                'updated_by' => $sourceId,
                                'activity' => Purchase::PURCHASE_DELETE_ACTIVITY,
                                'activity_slug' =>Str::of(Purchase::PURCHASE_DELETE_ACTIVITY)->slug('_'),
                                'target_name' => $targetName,
                                'details' => $activityDetails,
                            ];
                            ActivityLog::log($data);
                        }

                        // Now delete the purchase and its products
                        $purchase->productDetails()->delete();
                        $purchase->delete();
                    }
                }
                $this->setResponse(false, 'Purchase records deleted successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }
}
