<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Support\Facades\Validator;
use App\Models\Conversation;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use App\Http\Resources\ConversationResource;

class MessageController extends Controller
{
    public function storeConversation(Request $request)
    {
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('message_add')) {

        $validator = Validator::make($request->all(), [
            'subject' => 'required|string|max:255',
            "attachment" => "filled|mimes:jpg,bmp,png,jpeg,svg|max:2048",
            'message_comment' => 'nullable|string',
            'sms_text' => 'nullable|string',
            'send_sms' => 'nullable|boolean',
            'send_mail' => 'nullable|boolean',
            'role' => 'required|string',
            'user' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        $role = $request->input('role');
        $userIds = explode(',', $request->input('user'));

        if ($role == 'all' && $userIds[0] == 'all') {
            $users = User::role([ 'admin', 'employee', 'customer', 'supplier'])->pluck('id')->toArray();
        } else {
            if ($role != 'all' && $userIds[0] == 'all') {
                $users = User::role($role)->pluck('id')->toArray();
            } else {
                $users = $userIds;
            }
        };

        // Check if the owner's ID is in the members list, add it if not
        // $ownerId = auth()->id();
        // if (!in_array($ownerId, $users)) {
        //     $users[] = $ownerId;
        // }

        $conversation = auth()->user()->sentConversations()->create($request->except('attachment'));

        if ($request->hasFile('attachment')) {
            $image = $this->uploadFile($request->attachment, 'attachments/message');
            if ($image != false) {
                $conversation->attachment = $image;
            }
        }

        $conversation->save();

        // Add conversation members
        $conversation->members()->createMany(array_map(function ($user) {
            return ['user_id' => $user];
        }, $users));

        $message = $conversation->messages()->create($request->except('attachment'));

        if ($request->hasFile('attachment')) {
            $image = $this->uploadFile($request->attachment, 'attachments/message');
            if ($image != false) {
                $message->attachment = $image;
            }
            $message->save();
        }

        return response()->json(['message' => 'Message sent successfully', 'conversation_id' => $conversation->id, 'message_id' => $message->id], 201);
    } else {
        $data = [
            'error' => true,
            'message' => 'Unauthorized',
            'error_code' => 401
        ];
        return response()->json($data, 401);
    }
    }

    public function storeReplyMessage(Request $request, $conversationId)
    {
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('message_add')) {
        $validator = Validator::make($request->all(), [
            'subject' => 'required|string|max:255',
            'attachment" => "filled|mimes:jpg,bmp,png,jpeg,svg|max:2048',
            'message_comment' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 400);
        }

        // Find the conversation
        $conversation = Conversation::findOrFail($conversationId);

        $message = $conversation->messages()->create($request->except('attachment'));

        if ($request->hasFile('attachment')) {
            $image = $this->uploadFile($request->attachment, 'attachments/message');
            if ($image != false) {
                $message->attachment = $image;
            }
        }
        $message->save();

        return response()->json(['message' => 'Reply message stored successfully', 'conversation_id' => $conversation->id, 'message_id' => $message->id], 201);
    } else {
        $data = [
            'error' => true,
            'message' => 'Unauthorized',
            'error_code' => 401
        ];
        return response()->json($data, 401);
    }
    }

    public function getSentMessages(Request $request)
    {
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('message_view')) {
            $user = auth()->user();

            // Get sent conversations with the latest message
            $conversations = $user->sentConversations()
                ->with(['messages' => function ($query) {
                    $query->latest(); // Fetch the latest message first
                }])
                ->get();

            // Sort conversations by the latest message timestamp
            $conversations = $conversations->sortByDesc(function ($conversation) {
                return $conversation->messages->first()->created_at;
            });

            $conversationData = $conversations->map(function ($conversation) use ($user) {
                $messageFor = $conversation->members->reject(function ($member) use ($user) {
                    return $member->user_id === $user->id; // Exclude the authenticated user
                })->pluck('user.full_name')->implode(', '); // Get full names and join them

                $conversation->message_for = $messageFor; // Assign message_for field
                return $conversation;
            })->unique('id');

            return response()->json(['conversations' => ConversationResource::collection($conversationData)]);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function getInboxMessages(Request $request)
    {
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('message_view')) {
            $user = auth()->user();

        // Get inbox conversations with the latest message
        $conversations = $user->inboxConversations()
            ->with(['messages' => function ($query) {
                $query->latest(); // Fetch the latest message first
            }])
            ->get();

        // Sort conversations by the latest message timestamp
        $conversations = $conversations->sortByDesc(function ($conversation) {
            return $conversation->messages->first()->created_at;
        });

            return response()->json(['conversations' => ConversationResource::collection($conversations)]);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function getFullConversation($conversationId)
    {
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('message_view')) {

            // Find the conversation
            $conversation = Conversation::with(['messages'])->findOrFail($conversationId);

            $loggedInUserId = auth()->id();

            $conversation->markMessagesAsReadForUser($loggedInUserId);

            $filteredMessages = $conversation->filteredMessagesForUser($loggedInUserId);

            $conversation->setRelation('messages', $filteredMessages);

            return new ConversationResource($conversation);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function deleteConversation(Request $request, $conversationId)
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('message_delete')) {
            $conversation = Conversation::findOrFail($conversationId);

            $user = auth()->user();

            $isMember = $conversation->members()->where('user_id', $user->id)->exists();
            if (!$isMember) {
                return response()->json(['error' => 'You are not a member of this conversation'], 403);
            }

            $conversation->members()->where('user_id', $user->id)->delete();

            return response()->json(['message' => 'Conversation deleted successfully'], 200);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'Conversation not found'], 404);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function deleteMessage(Request $request, $conversationId, $messageId)
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('message_delete')) {
            $conversation = Conversation::findOrFail($conversationId);
            $message = $conversation->messages()->findOrFail($messageId);
            $user = auth()->user();
            $isOwner = $conversation->owner->id === $user->id;
            $isMember = $conversation->members()->where('user_id', $user->id)->exists();

            if (!$isOwner && !$isMember) {
                return response()->json(['error' => 'You are not authorized to delete this message'], 403);
            }

            $message->softDeleteMessage($user->id);

            return response()->json(['message' => 'Message deleted successfully'], 200);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'Message not found'], 404);
        } catch (\Exception $e) {
            return response()->json(['error' => $e], 500);
        }
    }

    public function deleteMultiConversation(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'conversation_ids' => 'required|array',
            'conversation_ids.*' => 'required|exists:conversations,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()->all()], 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('message_delete')) {
            $user = auth()->user();
            $userId = $user->id;

            foreach ($request->conversation_ids as $conversationId) {
                $conversation = Conversation::find($conversationId);

                if ($conversation) {
                    $isMember = $conversation->members()->where('user_id', $userId)->exists();

                    if (!$isMember) {
                        return response()->json(['error' => "You are not a member of conversation ID $conversationId"], 403);
                    }

                    $conversation->members()->where('user_id', $userId)->delete();
                }
            }

            return response()->json(['message' => 'Conversations deleted successfully'], 200);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'One or more conversations not found'], 404);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }

    public function deleteMultiMessage(Request $request, $conversationId)
    {
        $validator = Validator::make($request->all(), [
            'message_ids' => 'required|array',
            'message_ids.*' => 'required|exists:conversation_messages,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()->all()], 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('message_delete')) {
            $conversation = Conversation::findOrFail($conversationId);
            $user = auth()->user();
            $isOwner = $conversation->owner->id === $user->id;
            $isMember = $conversation->members()->where('user_id', $user->id)->exists();

            if (!$isOwner && !$isMember) {
                return response()->json(['error' => 'You are not authorized to delete messages in this conversation'], 403);
            }

            foreach ($request->message_ids as $messageId) {
                $message = $conversation->messages()->findOrFail($messageId);

                if ($message) {
                    $message->softDeleteMessage($user->id);
                }
            }

            return response()->json(['message' => 'Messages deleted successfully'], 200);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'Conversation or one or more messages not found'], 404);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Something went wrong'], 500);
        }
    }
}