<?php

namespace App\Http\Controllers;

use App\Http\Resources\IncomeListResource;
use App\Http\Resources\IncomeMonthlyReportResource;
use App\Models\Income;
use App\Models\ActivityLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;

class IncomeController extends Controller
{

    public function index()
    {
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('income_view')) {
            if (auth()->user()->hasPermissionTo('income_own_data')) {
                $incomes = Income::where(['customer_id' => auth()->user()->id])->get();
            } else {
                $incomes = Income::all();
            }
            return IncomeListResource::collection($incomes)->additional(["error" => false, "message" => '']);
        } else {
            $data = [ 
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:users,id',
            'status_id' => 'required|exists:categories,id',
            'date' => 'required|date',
            'label' => 'required',
            'entries' => 'required|json'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('income_add')) {

             $income=Income::create([
                "customer_id" => $request->customer_id,
                "status_id" => $request->status_id,
                "date" => $request->date,
                "label" => $request->label,
                "entries" => $request->entries,
            ]);

            $sourceName = auth()->id();
            $users = User::find($sourceName);
            $sourceId = $users ? $users->first_name : 'Unknown User';
        $data = [
            'updated_by' => $sourceId,
            'activity' => Income::INCOME_CREATE_ACTIVITY,
            'activity_slug' =>Str::of(Income::INCOME_CREATE_ACTIVITY)->slug('_'),
            'target_name' => $income->label,
            'details' => "Income for user '{$income->customer->first_name}' created",
        ];
        ActivityLog::log($data);
            $this->setResponse(false, 'Income created successfully');
            return response()->json($this->_response, 200);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized', 
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'income_id' => 'required|exists:incomes,id',
            'customer_id' => 'required|exists:users,id',
            'status_id' => 'required|exists:categories,id',
            'date' => 'required|date',
            'label' => 'required',
            'entries' => 'required|json'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('income_edit')) {

            $income = Income::find($request->income_id);
            $income->customer_id = $request->customer_id;
            $income->status_id = $request->status_id;
            $income->date = $request->date;
            $income->label = $request->label;
            $income->entries = $request->entries;
            $income->save();
            $sourceName = auth()->id();
            $users = User::find($sourceName);
            $sourceId = $users ? $users->first_name : 'Unknown User';
        $data = [
            'updated_by' => $sourceId,
            'activity' => Income::INCOME_UPDATE_ACTIVITY,
            'activity_slug' =>Str::of(Income::INCOME_UPDATE_ACTIVITY)->slug('_'),
            'target_name' => $income->label,
            'details' => "Income for user '{$income->customer->first_name}' updated",
        ];
        ActivityLog::log($data);
            $this->setResponse(false, 'Income updated successfully');
            return response()->json($this->_response, 200);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function destroy(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'income_ids' => 'required|array',
            'income_ids.*' => 'required|exists:incomes,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('income_delete')) {

            $deletedIncomes = Income::whereIn('id', $request->income_ids)->get();

            Income::whereIn('id', $request->income_ids)->delete();
            
            $sourceName = auth()->id();
            $users = User::find($sourceName);
            $sourceId = $users ? $users->first_name : 'Unknown User';
            foreach ($deletedIncomes as $income) {
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => Income::INCOME_DELETE_ACTIVITY,
                    'activity_slug' =>Str::of(Income::INCOME_DELETE_ACTIVITY)->slug('_'),
                    'target_name' => $income->label,
                    'details' => "Income for user '{$income->customer->first_name}' deleted",
                ];
                ActivityLog::log($data);
            }

            $this->setResponse(false, 'Income deleted successfully');
            return response()->json($this->_response, 200);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    public function monthlyReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date',
            'end_date' => 'required|date',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('income_view')) {

            $startDate = date('Y-m-d', strtotime($request->start_date));
            $endDate = date('Y-m-d', strtotime($request->end_date));

            $incomeData = Income::whereDate('date', '>=', $startDate)->whereDate('date', '<=', $endDate)->get();
            return IncomeMonthlyReportResource::collection($incomeData)->additional(["error" => false, "message" => '']);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }
}
