<?php

namespace App\Http\Controllers;

use App\Events\SendFcmNotification;
use App\Models\Complaint;
use App\Models\Notifications;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Mail\ComplainStatus;
use App\Http\Resources\ComplaintResource;
use App\Models\User;
use App\Models\ActivityLog;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;

class ComplaintController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('complaint_view')) {
                if (auth()->user()->hasPermissionTo('complaint_own_data')) {
                    if (auth()->user()->getRole() == 'customer') {
                        $complaints = Complaint::where(['customer_id' => auth()->user()->id])->Orderby('created_at', 'desc')->get();
                    } elseif (auth()->user()->getRole() == 'employee') {
                        $complaints = Complaint::where(['assign_to_id' => auth()->user()->id])->Orderby('created_at', 'desc')->get();
                    }
                } else {
                    $complaints = Complaint::Orderby('created_at', 'desc')->get();
                }

                return ComplaintResource::collection($complaints)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (auth()->user()->getRole() == 'customer') {
            $request->request->add(['customer_id' => auth()->user()->id]);
        }
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:users,id',
            'date' => 'required|date',
            'complaint_time' => 'required',
            'status_id' => 'required|exists:categories,id',
            'complaint_type_id' => 'required|exists:categories,id',
            'description' => 'nullable',
            'product_id' => 'required|exists:products,id',
            'assign_to_id' => 'required|exists:users,id',
            'assign_date' => 'required|date',
            'attachment' => 'filled|mimes:jpg,bmp,png,jpeg,svg,doc,docx,pdf,csv,txt|max:2048',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('complaint_add')) {

                $complaint = Complaint::create($request->all());

                if ($request->has('attachment')) {

                    $attachment = $this->uploadFile($request->attachment, 'images/complaint');
                    if ($attachment != false) {
                        $complaint->attachment = $attachment;
                    }
                }
                $complaint->save();

                // Complaint number assigned - mail to customer
                $systemName = getSetting('system_name');
                // Get email template content using helper
                $emailTemplate = getEmailContent('complaint_customer');
                $logo = getSetting('logo');
                $logo_image = url('storage/' . $logo);
                $footer_image = url('storage/email_image/footer.png');
                $support_image = url('storage/email_image/support.png');
                $service_image = url('storage/email_image/service.png');
                $history_image = url('storage/email_image/history.png');

                if ($emailTemplate) {
                    $mailformat = $emailTemplate->body;
                    $serch = array('{ username }', '{ complaint_number }', '{ systemname }');
                    $replace = array($complaint->customer->full_name, $complaint->complaint_no, $systemName);
                    $message_content = str_replace($serch, $replace, $mailformat);

                    // Welcome Mail
                    $mailData = [
                        'host' => getHost(),
                        'subject' => $emailTemplate->subject,
                        'content' => $message_content,
                        'logo_image' => $logo_image,
                        'footer_image' => $footer_image,
                        'support_image' => $support_image,
                        'service_image' => $service_image,
                        'history_image' => $history_image
                    ];

                    Mail::to($complaint->customer->email)->send(new ComplainStatus($mailData));
                }

                // New complain notification - mail to administrator
                $systemName = getSetting('system_name');
                $systemEmail = getSetting('email');
                $admin = User::role(['admin'])->get();
                $adminName = $admin->first()->full_name;
                // Get email template content using helper
                $emailTemplate = getEmailContent('complaint_admin');
                $logo = getSetting('logo');
                $logo_image = url('storage/' . $logo);
                $footer_image = url('storage/email_image/footer.png');
                $support_image = url('storage/email_image/support.png');
                $service_image = url('storage/email_image/service.png');
                $history_image = url('storage/email_image/history.png');

                if ($emailTemplate) {
                    $mailformat = $emailTemplate->body;
                    $serch = array('{ admin }', '{ customer }', '{ complaint_number }', '{ complaint_date }', '{ systemname }');
                    $replace = array($adminName, $complaint->customer->full_name, $complaint->complaint_no, $complaint->date, $systemName);
                    $message_content = str_replace($serch, $replace, $mailformat);

                    // Welcome Mail
                    $mailData = [
                        'host' => getHost(),
                        'subject' => $emailTemplate->subject,
                        'content' => $message_content,
                        'logo_image' => $logo_image,
                        'footer_image' => $footer_image,
                        'support_image' => $support_image,
                        'service_image' => $service_image,
                        'history_image' => $history_image
                    ];

                    Mail::to($systemEmail)->send(new ComplainStatus($mailData));
                }


                // New Complaint assigned -  mail to employee
                $systemName = getSetting('system_name');
                // Get email template content using helper
                $emailTemplate = getEmailContent('complaint_employee');
                $logo = getSetting('logo');
                $logo_image = url('storage/' . $logo);
                $footer_image = url('storage/email_image/footer.png');
                $support_image = url('storage/email_image/support.png');
                $service_image = url('storage/email_image/service.png');
                $history_image = url('storage/email_image/history.png');

                if ($emailTemplate) {
                    $mailformat = $emailTemplate->body;
                    $serch = array('{ employee }', '{ customer }', '{ complaint_number }', '{ complaint_date }', '{ systemname }');
                    $replace = array($complaint->assignTo->full_name, $complaint->customer->full_name, $complaint->complaint_no, date("d-m-Y"), $systemName);
                    $message_content = str_replace($serch, $replace, $mailformat);

                    // Welcome Mail
                    $mailData = [
                        'host' => getHost(),
                        'subject' => $emailTemplate->subject,
                        'content' => $message_content,
                        'logo_image' => $logo_image,
                        'footer_image' => $footer_image,
                        'support_image' => $support_image,
                        'service_image' => $service_image,
                        'history_image' => $history_image
                    ];

                    Mail::to($complaint->assignTo->email)->send(new ComplainStatus($mailData));
                }

                $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                $targetName =  $complaint->product->name;
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => Complaint::COMPLAINT_CREATE_ACTIVITY,
                    'activity_slug' =>Str::of(Complaint::COMPLAINT_CREATE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => "Complaint for product '{$targetName}' created",
                ];
            ActivityLog::log($data);
            $notifications = [
                [
                    'user_id' => $complaint->assignTo->id,
                    'target_id' => $complaint->id,
                    'updated_by' => $sourceId,
                    'content' => "New Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} assigned to you.",
                    'is_read' => 0, // Set as unread (0) by default
                    'path' => Complaint::COMPLAINT_PATH
                ],
                [
                    'user_id' => $complaint->customer_id , // Change this to the appropriate user ID
                    'target_id' => $complaint->id,
                    'updated_by' => $sourceId,
                    'content' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been received.",
                    'is_read' => 0, // Set as unread (0) by default
                    'path' => Complaint::COMPLAINT_PATH
                ],
                [
                    'user_id' => $admin->first()->id , // Change this to the appropriate user ID
                    'target_id' => $complaint->id,
                    'updated_by' => $sourceId,
                    'content' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been received.",
                    'is_read' => 0, // Set as unread (0) by default
                    'path' => Complaint::COMPLAINT_PATH
                ],
            ];
            
            // Send multiple notifications
            Notifications::send($notifications);
            $notificationData = [
                [
                    'deviceToken' => $complaint->assignTo->device_token,
                    'title' => 'New Complaint Assigned',
                    'body' => "New Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} assigned to you.",
                ],
                [
                    'deviceToken' => $complaint->customer->device_token,
                    'title' => 'Complaint Received',
                    'body' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been received.",
                ],
                [
                    'deviceToken' => $admin->first()->device_token,
                    'title' => 'New Complaint Received',
                    'body' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been received.",
                ],
            ];
    
            // Trigger event to send FCM notifications
            event(new SendFcmNotification($notificationData));
                $this->_response['data'] = '';
                $this->setResponse(false, 'Complaint created successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Sale  $sale
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:complaints,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('complaint_view')) {

                $complaint = Complaint::find($request->id);

                return (new ComplaintResource($complaint))->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Complaint  $complaint
     * @return \Illuminate\Http\Response
     */
    public function edit(Complaint $complaint)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Quotation  $quotation
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        if (auth()->user()->getRole() == 'customer') {
            $request->request->add(['customer_id' => auth()->user()->id]);
        }

        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:complaints,id',
            'customer_id' => 'required|exists:users,id',
            'complaint_time' => 'required',
            'date' => 'required',
            'status_id' => 'required|exists:categories,id',
            'complaint_type_id' => 'required|exists:categories,id',
            'description' => 'nullable',
            'product_id' => 'required|exists:products,id',
            'assign_to_id' => 'required|exists:users,id',
            'assign_date' => 'required',
            'attachment' => 'filled|mimes:jpg,bmp,png,jpeg,svg,doc,docx,pdf,csv,txt|max:2048',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('complaint_edit')) {

            $complaint = Complaint::find($request->id);
            $old_status_id = $complaint->status_id;
            $complaint->update($request->all());

            if ($request->has('attachment')) {

                $attachment = $this->uploadFile($request->attachment, 'images/complaint');
                if ($attachment != false) {
                    $complaint->attachment = $attachment;
                }
            }
            $complaint->save();
            $admin = User::role(['admin'])->get();
            $sourceName = auth()->id();
            $user = User::find($sourceName);
            $sourceId = $user ? $user->first_name : 'Unknown User';
            if ($old_status_id != $request->status_id) {
                if(auth()->user()->getRole() == 'admin'){
                    switch ($request->status_id) {
                        case 28: // Complaint is open
                            $notifications = [ [
                                'user_id' => $complaint->customer_id,
                                'target_id' => $complaint->id,
                                'updated_by' => $sourceId,
                                'content' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as opened.",
                                'is_read' => 0,
                                'path' => Complaint::COMPLAINT_PATH
                            ],
                            [
                                'user_id' => $complaint->assign_to_id,
                                'target_id' => $complaint->id,
                                'updated_by' => $sourceId,
                                'content' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as opened.",
                                'is_read' => 0,
                                'path' => Complaint::COMPLAINT_PATH
                            ],
                            ];
                            Notifications::send($notifications);
                            $notificationData = [
                                [
                                    'deviceToken' => $complaint->customer->device_token,
                                    'title' => 'Complaint Opened',
                                    'body' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as opened.",
                                ],
                                [
                                    'deviceToken' => $complaint->assign_to->device_token,
                                    'title' => 'Complaint Opened',
                                    'body' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as opened.",
                                ]
                                ];
                                event(new SendFcmNotification($notificationData));
                            break;
                        case 29: // Complaint is in progress
                            $notifications = [ [
                                'user_id' => $complaint->customer_id,
                                'target_id' => $complaint->id,
                                'updated_by' => $sourceId,
                                'content' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as in progress.",
                                'is_read' => 0,
                                'path' => Complaint::COMPLAINT_PATH
                            ],
                            [
                                'user_id' => $complaint->assign_to_id,
                                'target_id' => $complaint->id,
                                'updated_by' => $sourceId,
                                'content' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as in progress.",
                                'is_read' => 0,
                                'path' => Complaint::COMPLAINT_PATH
                            ],
                            ];
                            Notifications::send($notifications);
                            $notificationData = [
                                [
                                    'deviceToken' => $complaint->customer->device_token,
                                    'title' => 'Complaint In Progress',
                                    'body' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as in progress.",
                                ],
                                [
                                    'deviceToken' => $complaint->assign_to->device_token,
                                    'title' => 'Complaint In Progress',
                                    'body' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as in progress.",
                                ]
                            ];
                            event(new SendFcmNotification($notificationData));
                            break;
                        case 30: // Complaint is closed and solved
                            $notifications = [ [
                                'user_id' => $complaint->customer_id,
                                'target_id' => $complaint->id,
                                'updated_by' => $sourceId,
                                'content' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been closed.",
                                'is_read' => 0,
                                'path' => Complaint::COMPLAINT_PATH
                            ],
                            [
                                'user_id' => $complaint->assign_to_id,
                                'target_id' => $complaint->id,
                                'updated_by' => $sourceId,
                                'content' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as closed and solved.",
                                'is_read' => 0,
                                'path' => Complaint::COMPLAINT_PATH
                            ],
                            ];
                            Notifications::send($notifications);
                            $notificationData = [
                                [
                                    'deviceToken' => $complaint->customer->device_token,
                                    'title' => 'Complaint Closed',
                                    'body' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been closed.",
                                ],
                                [
                                    'deviceToken' => $complaint->assign_to->device_token,
                                    'title' => 'Complaint Closed',
                                    'body' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as closed and solved.",
                                ]
                            ];
                            event(new SendFcmNotification($notificationData));
                            break;
                        // Add more cases for other status changes if needed
                        default:
                            // No action for other status changes
                            break;
                    }
                }
                else{
                switch ($request->status_id) {
                    case 28: // Complaint is open
                        $notifications = [ [
                            'user_id' => $complaint->customer_id,
                            'target_id' => $complaint->id,
                            'updated_by' => $sourceId,
                            'content' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as opened.",
                            'is_read' => 0,
                            'path' => Complaint::COMPLAINT_PATH
                        ],
                        [
                            'user_id' => $complaint->assign_to_id,
                            'target_id' => $complaint->id,
                            'updated_by' => $sourceId,
                            'content' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as opened.",
                            'is_read' => 0,
                            'path' => Complaint::COMPLAINT_PATH
                        ],
                        [
                            'user_id' => $admin->first()->id,
                            'target_id' => $complaint->id,
                            'updated_by' => $sourceId,
                            'content' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as opened.",
                            'is_read' => 0,
                            'path' => Complaint::COMPLAINT_PATH
                        ]];
                        Notifications::send($notifications);
                        $notificationData = [
                            [
                                'deviceToken' => $complaint->customer->device_token,
                                'title' => 'Complaint Opened',
                                'body' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as opened.",
                            ],
                            [
                                'deviceToken' => $complaint->assign_to->device_token,
                                'title' => 'Complaint Opened',
                                'body' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as opened.",
                                ],
                                [
                                'deviceToken' => $admin->first()->device_token,
                                'title' => 'Complaint Opened',
                                'body' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as opened.",
                            ]
                                ];
                                event(new SendFcmNotification($notificationData));
                        break;
                    case 29: // Complaint is in progress
                        $notifications = [ [
                            'user_id' => $complaint->customer_id,
                            'target_id' => $complaint->id,
                            'updated_by' => $sourceId,
                            'content' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as in progress.",
                            'is_read' => 0,
                            'path' => Complaint::COMPLAINT_PATH
                        ],
                        [
                            'user_id' => $complaint->assign_to_id,
                            'target_id' => $complaint->id,
                            'updated_by' => $sourceId,
                            'content' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as in progress.",
                            'is_read' => 0,
                            'path' => Complaint::COMPLAINT_PATH
                        ],
                        [
                            'user_id' => $admin->first()->id,
                            'target_id' => $complaint->id,
                            'updated_by' => $sourceId,
                            'content' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as in progress.",
                            'is_read' => 0,
                            'path' => Complaint::COMPLAINT_PATH
                        ]];
                        Notifications::send($notifications);
                        $notificationData = [
                            [
                                'deviceToken' => $complaint->customer->device_token,
                                'title' => 'Complaint In Progress',
                                'body' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as in progress.",
                            ],
                            [
                                'deviceToken' => $complaint->assign_to->device_token,
                                'title' => 'Complaint In Progress',
                                'body' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as in progress.",
                            ],
                            [
                                'deviceToken' => $admin->first()->device_token,
                                'title' => 'Complaint In Progress',
                                'body' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as in progress.",
                            ]
                            ];
                            event(new SendFcmNotification($notificationData));
                        break;
                    case 30: // Complaint is closed and solved
                        $notifications = [ [
                            'user_id' => $complaint->customer_id,
                            'target_id' => $complaint->id,
                            'updated_by' => $sourceId,
                            'content' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been closed.",
                            'is_read' => 0,
                            'path' => Complaint::COMPLAINT_PATH
                        ],
                        [
                            'user_id' => $complaint->assign_to_id,
                            'target_id' => $complaint->id,
                            'updated_by' => $sourceId,
                            'content' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as closed and solved.",
                            'is_read' => 0,
                            'path' => Complaint::COMPLAINT_PATH
                        ],
                        [
                            'user_id' => $admin->first()->id,
                            'target_id' => $complaint->id,
                            'updated_by' => $sourceId,
                            'content' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as closed and solved.",
                            'is_read' => 0,
                            'path' => Complaint::COMPLAINT_PATH
                        ]];
                        Notifications::send($notifications);
                        $notificationData = [
                            [
                                'deviceToken' => $complaint->customer->device_token,
                                'title' => 'Complaint Closed',
                                'body' => "Your complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been closed.",
                            ],
                            [
                                'deviceToken' => $complaint->assign_to->device_token,
                                'title' => 'Complaint Closed',
                                'body' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as closed and solved.",
                            ],
                            [
                                'deviceToken' => $admin->first()->device_token,
                                'title' => 'Complaint Closed',
                                'body' => "Complaint {$complaint->complaint_no} for {$complaint->complaintType->value} {$complaint->product->name} has been marked as closed and solved.",
                            ]
                            ];
                            event(new SendFcmNotification($notificationData));
                        break;
                    // Add more cases for other status changes if needed
                    default:
                        // No action for other status changes
                        break;
                }
            };
                $systemName = getSetting('system_name');
                $statusLabel = categoryLabel($complaint->status_id);
                // Get email template content using helper
                $emailTemplate = getEmailContent('complaint_resolve_customer');
                $logo = getSetting('logo');
                $logo_image = url('storage/' . $logo);
                $footer_image = url('storage/email_image/footer.png');
                $support_image = url('storage/email_image/support.png');
                $service_image = url('storage/email_image/service.png');
                $history_image = url('storage/email_image/history.png');

                if ($emailTemplate) {
                    $mailformat = $emailTemplate->body;
                    $serch = array('{ username }', '{ complaint_number }', '{ complaint_status }', '{ complaint_date }', '{ systemname }');
                    $replace = array($complaint->customer->full_name, $complaint->complaint_no, $statusLabel, date("d-m-Y"), $systemName);
                    $message_content = str_replace($serch, $replace, $mailformat);

                    // Welcome Mail
                    $mailData = [
                        'host' => getHost(),
                        'subject' => $emailTemplate->subject,
                        'content' => $message_content,
                        'logo_image' => $logo_image,
                        'footer_image' => $footer_image,
                        'support_image' => $support_image,
                        'service_image' => $service_image,
                        'history_image' => $history_image
                    ];

                    Mail::to($complaint->customer->email)->send(new ComplainStatus($mailData));
                }
            }

            $targetName = $complaint->product->name;
            $data = [
                'updated_by' => $sourceId,
                'activity' => Complaint::COMPLAINT_UPDATE_ACTIVITY,
                'activity_slug' =>Str::of(Complaint::COMPLAINT_UPDATE_ACTIVITY)->slug('_'),
                'target_name' => $targetName,
                'details' => "Complaint for product '{$targetName}' updated",
            ];
            ActivityLog::log($data);
            $this->_response['data'] = '';
            $this->setResponse(false, 'Complaint updated successfully.');
            return response()->json($this->_response);
        } else {
            $data = [
                'error' => true,
                'message' => 'Unauthorized',
                'error_code' => 401
            ];
            return response()->json($data, 401);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Product  $product
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:complaints,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('complaint_delete')) {

                $complaint =  Complaint::find($request->id);
                if ($complaint) {
                    $complaint->delete();

                    $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                    $targetName = $complaint->product->name;
                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Complaint::COMPLAINT_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(Complaint::COMPLAINT_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => "Complaint for product '{$targetName}' deleted",
                    ];
                    ActivityLog::log($data);
                    $this->setResponse(false, 'Complaint deleted successfully.');
                    return response()->json($this->_response);
                }
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete multiple AMC Data*/
    public function deleteMultiComplaint(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'complaint_ids' => 'required|array',
            'complaint_ids.*' => 'required|exists:complaints,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('complaint_delete')) {

                foreach ($request->complaint_ids as $id) {
                    $complaint =  Complaint::find($id);
                    if ($complaint) {
                        $complaint->delete();
                    }
                }

                $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                $targetName = $complaint->product->name;
                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => Complaint::COMPLAINT_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(Complaint::COMPLAINT_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => "Complaint for product '{$targetName}' deleted",
                    ];
                    ActivityLog::log($data);
                $this->setResponse(false, 'Complaints deleted successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }
}
