<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Amc;
use App\Models\Product;
use App\Models\AmcService;
use App\Models\User;
use App\Models\ActivityLog;
use App\Models\ServiceProduct;
use App\Models\Notifications;
use Illuminate\Http\Request;
use App\Mail\ServiceStatus;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use App\Http\Resources\AmcServiceResource;
use App\Events\SendFcmNotification;


class AmcServiceController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('service_view')) {
                if (auth()->user()->hasPermissionTo('service_own_data')) {
                    if (auth()->user()->getRole() == 'customer') {
                        $services = AmcService::whereHas('amc', function ($q) {
                            $q->where('customer', auth()->user()->id);
                        })->Orderby('created_at', 'desc')->get();
                    } elseif (auth()->user()->getRole() == 'employee') {
                        $services = AmcService::where(['assign_to' => auth()->user()->id])->Orderby('created_at', 'desc')->get();
                    }
                } else {
                    $services = AmcService::Orderby('created_at', 'desc')->get();
                }

                return AmcServiceResource::collection($services)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amc_id' => 'required|exists:amcs,id',
            'service_date' => 'required|date',
            'service_time' => 'required',
            'status' => 'required|exists:categories,id',
            'assign_to' => 'required|exists:users,id',
            'is_chargeble' => 'required|boolean',
            'charge' => 'required_if:is_chargeble,true|numeric|',
            'service_product' => 'required|array',
            'service_product.*' => 'required|exists:products,id',
            'service_product_note' => 'required|array',
            'before_service_img' => 'filled|mimes:jpg,bmp,png,jpeg,svg|max:2048',
            'after_service_img' => 'filled|mimes:jpg,bmp,png,jpeg,svg|max:2048',
            'video' => 'filled|file|mimes:mp4,mov,avi|max:20480', // Adjust max file size as needed
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('service_add')) {

                $exist = AmcService::where(['service_date' => Carbon::parse($request->service_date)->format('d-m-Y'), 'amc_id' => $request->amc_id])->count();
                if ($exist) {
                    $this->setResponse(false, "Service already added for this AMC and date.");
                    return response()->json($this->_response, 400);
                }

                $service = AmcService::create($request->except('before_service_img,after_service_img,video'));
                if ($request->has('before_service_img')) {

                    $image = $this->uploadFile($request->before_service_img, 'images/service');
                    if ($image != false) {
                        $service->before_service_img = $image;
                    }
                }
                if ($request->has('after_service_img')) {

                    $image = $this->uploadFile($request->after_service_img, 'images/service');
                    if ($image != false) {
                        $service->after_service_img = $image;
                    }
                }
                if ($request->has('video')) {

                    $image = $this->uploadFile($request->video, 'videos/service');
                    if ($image != false) {
                        $service->video = $image;
                    }
                }
                
                // Service Selected Products
                foreach ($request->service_product as $key => $value) {
                    $service->products()->create([
                        'service_product' => $request->service_product[$key],
                        'product_note' => $request->service_product_note[$key],
                    ]);
                }
                $service->save();
                $sourceName = auth()->id();
                $user = User::find($sourceName);
                $sourceId = $user ? $user->first_name : 'Unknown User';
                $product = Product::find($request->service_product[$key]);
                $targetName = $product->name;
    
                $activityDetails = "Service for product '{$targetName}' created";
                // $activityDetails = "Created service no. {$service->service_no}";
    
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => AmcService::SERVICE_CREATE_ACTIVITY,
                    'activity_slug' =>Str::of(AmcService::SERVICE_CREATE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => $activityDetails,
                ];
                ActivityLog::log($data);
                $admin = User::role(['admin'])->get();
                if (!$admin) {
                    throw new \Exception('Admin user not found.');
                }
                $notifications = [
                    [
                        'user_id' => $service->employee->id , // Change this to the appropriate user ID
                        'target_id' => $service->id,
                        'updated_by' => $sourceId,
                        'content' => "New AMC Service Request  {$service->service_no} Assigned to You.",
                        'is_read' => 0, // Set as unread (0) by default
                        'path' => AmcService::SERVICE_PATH, // Set the path to the activity
                    ],
                    [
                        'user_id' => $service->amc->client->id , // Change this to the appropriate user ID
                        'target_id' => $service->id,
                        'updated_by' => $sourceId,
                        'content' => "Thank you for your AMC Service Request. We are on it!",
                        'is_read' => 0, // Set as unread (0) by default
                        'path' => AmcService::SERVICE_PATH, // Set the path to the activity
                    ],
                    [
                        'user_id' => $admin->first()->id , // Change this to the appropriate user ID
                        'target_id' => $service->id,
                        'updated_by' => $sourceId,
                        'content' => "A new service request  {$service->service_no} has been received.",
                        'is_read' => 0, // Set as unread (0) by default
                        'path' => AmcService::SERVICE_PATH, // Set the path to the activity
                    ],
                ];
                
                // Send multiple notifications
                Notifications::send($notifications);
                // $deviceToken = $service->employee->device_token;
                $notificationData = [
                    [
                        'deviceToken' => $service->employee->device_token,
                        'title' => 'New AMC Service Request',
                        'body' => "New AMC Service Request {$service->service_no} Assigned to You.",
                    ],
                    [
                        'deviceToken' => $service->amc->client->device_token,
                        'title' => 'AMC Service Request Update',
                        'body' => 'Thank you for your AMC Service Request. We are on it!',
                    ],
                    [
                        'deviceToken' => $admin->first()->device_token,
                        'title' => 'New Service Request Received',
                        'body' => "A new service request {$service->service_no} has been received.",
                    ],
                ];
                    event(new SendFcmNotification($notificationData));
                $this->_response['data'] = '';
                $this->setResponse(false, 'Service created successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amc_services,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('service_view')) {

                $amc = AmcService::find($request->id);

                return (new AmcServiceResource($amc))->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amc_services,id',
            'amc_id' => 'required|exists:amcs,id',
            'service_date' => 'required|date',
            'service_time' => 'required',
            'status' => 'required|exists:categories,id',
            'assign_to' => 'required|exists:users,id',
            'is_chargeble' => 'required|boolean',
            'charge' => 'required_if:is_chargeble,true|numeric|',
            'service_product' => 'required|array',
            'service_product.*' => 'required|exists:products,id',
            'service_product_note' => 'required|array',
            'before_service_img' => 'filled|mimes:jpg,bmp,png,jpeg,svg|max:2048',
            'after_service_img' => 'filled|mimes:jpg,bmp,png,jpeg,svg|max:2048',
            'video' => 'filled|file|mimes:mp4,mov,avi|max:20480', // Adjust max file size as needed
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('service_edit')) {

                $service = AmcService::find($request->id);
                if ($request->status != $request->employee_status) {
                    $request->request->add(['is_employee_status_approved' => false]);
                }
                $old_employee_status_id = $service->employee_status;
                $service->update($request->except('before_service_img,after_service_img,video'));
                // Delete old records
                if ($request->has('before_service_img')) {

                    $image = $this->uploadFile($request->before_service_img, 'images/service');
                    if ($image != false) {
                        $service->before_service_img = $image;
                    }
                }
                if ($request->has('after_service_img')) {

                    $image = $this->uploadFile($request->after_service_img, 'images/service');
                    if ($image != false) {
                        $service->after_service_img = $image;
                    }
                }
                if ($request->has('video')) {

                    $image = $this->uploadFile($request->video, 'videos/service');
                    if ($image != false) {
                        $service->video = $image;
                    }
                }
                $service->products()->delete();

                // Service Selected Products
                foreach ($request->service_product as $key => $value) {
                    $service->products()->create([
                        'service_product' => $request->service_product[$key],
                        'product_note' => $request->service_product_note[$key],
                    ]);
                }
                $service->save();
                $admin = User::role(['admin'])->get();
                $sourceName = auth()->id();
                $user = User::find($sourceName);
                $sourceId = $user ? $user->first_name : 'Unknown User';
        if ($old_employee_status_id != $request->employee_status){
            $notificationData = [];
            switch ($request->employee_status) {
                case 17: // Open
                    $notifications = [
                        [
                            'user_id' => $admin->first()->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceId,
                            'content' => "AMC Service Request {$service->service_no} is now in open.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ]
                    ];
                    Notifications::send($notifications);
                    $notificationData = [
                    [
                        'deviceToken' => $admin->first()->device_token,
                        'title' => 'AMC Service Request Update',
                        'body' => "AMC Service Request {$service->service_no} is now in open.",
                    ]
                    ];
                    event(new SendFcmNotification($notificationData));
                    break;
                case 18: // In Progress
                    $notifications = [
                        [
                            'user_id' => $admin->first()->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceId,
                            'content' => "AMC Service Request {$service->service_no} is now in progress.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ]
                    ];
                    Notifications::send($notifications);
                    $notificationData = [
                        [
                        'deviceToken' => $admin->first()->device_token,
                        'title' => 'AMC Service Request Update',
                        'body' => "AMC Service Request {$service->service_no} is now in progress.",
                        ]
                    ];
                    event(new SendFcmNotification($notificationData));
                    break;
                case 19: // Completed
                    $notifications = [
                        [
                            'user_id' => $admin->first()->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceId,
                            'content' => "AMC Service Request {$service->service_no} has been completed.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ],
                    ];
                    Notifications::send($notifications);
                    $notificationData = [
                        [
                        'deviceToken' => $admin->first()->device_token,
                        'title' => 'AMC Service Request Update',
                        'body' => "AMC Service Request {$service->service_no} has been completed.",
                        ],
                    ];
                    event(new SendFcmNotification($notificationData));
                    break;
                // Add more cases for other status changes if needed
                default:
                    // No action for other status changes
                    break;
            }
        }
                $product = Product::find($request->service_product[$key]);
                $targetName = $product->name;
    
                $activityDetails = "Service for product '{$targetName}' updated";
    
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => AmcService::SERVICE_UPDATE_ACTIVITY,
                    'activity_slug' =>Str::of(AmcService::SERVICE_UPDATE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => $activityDetails,
                ];
                ActivityLog::log($data);
                $this->_response['data'] = '';
                $this->setResponse(false, 'Service updated successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amc_services,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('service_delete')) {

                $service =  AmcService::find($request->id);
                if ($service) {
                    $service->delete();
                    
                    $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                    $amcProduct = $service->products->first();
                    $targetName = $amcProduct ? $amcProduct->product->name : 'Unknown Product';

        
                    $activityDetails = "Service for product '{$targetName}' deleted";
        
                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => AmcService::SERVICE_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(AmcService::SERVICE_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);
                    $this->setResponse(false, 'Service deleted successfully.');
                    return response()->json($this->_response);
                }
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete multiple AMC Data*/
    public function deleteMultiService(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'service_ids' => 'required|array',
            'service_ids.*' => 'required|exists:amc_services,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('service_delete')) {

                foreach ($request->service_ids as $id) {
                    $service =  AmcService::find($id);
                    if ($service) {
                        $service->delete();
                    }
                }

                    $sourceName = auth()->id();
                    $user = User::find($sourceName);
                    $sourceId = $user ? $user->first_name : 'Unknown User';
                    // $product = Product::find($request->service_product[$key]);
                    $targetName = $service->products->first()->name ?? 'Unknown Product';

        
                    $activityDetails = "Service for product '{$targetName}' deleted";
        
                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => AmcService::SERVICE_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(AmcService::SERVICE_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);
                $this->setResponse(false, 'Services deleted successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function rejectEmployeeStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amc_services,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $service =  AmcService::find($request->id);
            if ($service) {
                $service->update(['is_employee_status_approved' => true]);
                $sourceName = auth()->user()->first_name ?? 'Unknown User';

                // Prepare notifications based on AMC status
                $notifications = [];
                $notificationData = [];
    
                switch ($service->status) {
                    case 17: // Open
                        $notifications[] = [
                            'user_id' => $service->amc->client->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceName,
                            'content' => "AMC Service open status for {$service->service_no} has been rejected.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ];
                        $notificationData[] = [
                            'deviceToken' => $service->amc->client->device_token,
                            'title' => 'AMC Service Update',
                            'body' => "AMC Service open status for {$service->service_no} has been rejected.",
                        ];
                        break;
                    case 18: // In Progress
                        $notifications[] = [
                            'user_id' => $service->employee->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceName,
                            'content' => "AMC Service in progress status for {$service->service_no} has been rejected.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ];
                        $notificationData[] = [
                            'deviceToken' => $service->employee->device_token,
                            'title' => 'AMC Service Update',
                            'body' => "AMC Service in progress status for {$service->service_no} has been rejected.",
                        ];
                        break;
                    case 19: // Closed
                        $notifications[] = [
                            'user_id' => $service->employee->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceName,
                            'content' => "AMC Service closed status for {$service->service_no} has been rejected.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ];
                        $notificationData[] = [
                            'deviceToken' => $service->employee->device_token,
                            'title' => 'AMC Service Update',
                            'body' => "AMC Service closed status for {$service->service_no} has been rejected.",
                        ];
                        break;
                    default:
                        // Handle other status codes if needed
                        break;
                }
    
                // Send the prepared notifications
                Notifications::send($notifications);
                event(new SendFcmNotification($notificationData));
                $this->setResponse(false, 'Status rejected successfully.');
                return response()->json($this->_response);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function approveEmployeeStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amc_services,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $service =  AmcService::find($request->id);
            if ($service) {
                $old_status_id = $service->status;
                $service->update(['status' => $service->employee_status, 'is_employee_status_approved' => true]);

                // Service status mail to customer

                $systemName = getSetting('system_name');
                $statusLabel = categoryLabel($service->status);
                // Get email template content using helper
                $emailTemplate = getEmailContent('service_done_mail');
                $logo = getSetting('logo');
                $logo_image = url('storage/' . $logo);
                $footer_image = url('storage/email_image/footer.png');
                $support_image = url('storage/email_image/support.png');
                $service_image = url('storage/email_image/service.png');
                $history_image = url('storage/email_image/history.png');

                if ($emailTemplate) {
                    $mailformat = $emailTemplate->body;
                    $serch = array('{ username }', '{ service_title }', '{ service_status }', '{ service_date }', '{ systemname }');
                    $replace = array($service->amc->client->full_name, $service->service_label, $statusLabel, date("d-m-Y"), $systemName);
                    $message_content = str_replace($serch, $replace, $mailformat);

                    // Welcome Mail
                    $mailData = [
                        'host' => getHost(),
                        'subject' => $emailTemplate->subject,
                        'content' => $message_content,
                        'logo_image' => $logo_image,
                        'footer_image' => $footer_image,
                        'support_image' => $support_image,
                        'service_image' => $service_image,
                        'history_image' => $history_image
                    ];

                    Mail::to($service->amc->client->email)->send(new ServiceStatus($mailData));
                }
                $sourceName = auth()->user()->first_name ?? 'Unknown User';

                // Prepare notifications based on AMC status
                $notifications = [];
                $notificationData = [];
    
                switch ($service->status) {
                    case 17: // Open
                        $notifications[] = [
                            'user_id' => $service->amc->client->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceName,
                            'content' => "AMC service Request {$service->service_no} has been opened.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ];
                        $notificationData[] = [
                            'deviceToken' => $service->amc->client->device_token,
                            'title' => 'AMC Service Update',
                            'body' => "AMC service Request {$service->service_no} has been opened.",
                        ];
                        break;
                    case 18: // In Progress
                        $notifications[] = [
                            'user_id' => $service->amc->client->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceName,
                            'content' => "AMC service Request {$service->service_no} is now in progress.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ];
                        $notifications[] = [
                            'user_id' => $service->employee->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceName,
                            'content' => "AMC Service in progress status for {$service->service_no} has been approved.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ];
                        $notificationData[] = [
                            'deviceToken' => $service->amc->client->device_token,
                            'title' => 'AMC Service Update',
                            'body' => "AMC service Request {$service->service_no} is now in progress.",
                        ];
                        $notificationData[] = [
                            'deviceToken' => $service->employee->device_token,
                            'title' => 'AMC Service Update',
                            'body' => "AMC Service in progress status for {$service->service_no} has been approved.",
                        ];
                        break;
                    case 19: // Closed
                        $notifications[] = [
                            'user_id' => $service->amc->client->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceName,
                            'content' => "AMC service Request {$service->service_no} has been closed.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ];
                        $notifications[] = [
                            'user_id' => $service->employee->id,
                            'target_id' => $service->id,
                            'updated_by' => $sourceName,
                            'content' => "AMC Service closed status for {$service->service_no} has been approved.",
                            'is_read' => 0,
                            'path' => AmcService::SERVICE_PATH
                        ];
                        $notificationData[] = [
                            'deviceToken' => $service->amc->client->device_token,
                            'title' => 'AMC Service Update',
                            'body' => "AMC service Request {$service->service_no} has been closed.",
                        ];
                        $notificationData[] = [
                            'deviceToken' => $service->employee->device_token,
                            'title' => 'AMC Service Update',
                            'body' => "AMC Service closed status for {$service->service_no} has been approved.",
                        ];
                        break;
                    default:
                        // Handle other status codes if needed
                        break;
                }
    
                // Send the prepared notifications
                Notifications::send($notifications);
                event(new SendFcmNotification($notificationData));
                $this->setResponse(false, 'Status approved successfully.');
                return response()->json($this->_response);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }
}
