<?php

namespace App\Http\Controllers;

use App\Events\SendFcmNotification;
use App\Models\Amc;
use App\Models\Category;
use App\Models\ActivityLog;
use App\Models\Product;
use App\Models\AmcProduct;
use App\Models\User;

use App\Models\Notifications;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use App\Http\Resources\AmcResource;

class AmcController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('amc_view')) {
                if (auth()->user()->hasPermissionTo('amc_own_data')) {
                    if (auth()->user()->getRole() == 'customer') {
                        $amcs = Amc::where(['customer' => auth()->user()->id])->Orderby('created_at', 'desc')->get();
                    } elseif (auth()->user()->getRole() == 'employee') {
                        $amcs = Amc::where(['assign_to' => auth()->user()->id])->Orderby('created_at', 'desc')->get();
                    }
                } else {
                    $amcs = Amc::Orderby('created_at', 'desc')->get();
                }

                return AmcResource::collection($amcs)->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (UserNotDefinedException $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer' => 'required|exists:users,id',
            'date' => 'required|date',
            'contact_person' => 'nullable',
            'status' => 'required|exists:categories,id',
            'assign_to' => 'required|exists:users,id',
            'attachment' => 'filled|mimes:jpg,bmp,png,jpeg,svg,doc,docx,pdf,csv,txt|max:2048',
            // 'amc_detail' => 'required',
            'interval' => 'required|numeric|min:1',
            'number_of_service' => 'required|numeric|min:1',
            'service_date' => 'required|array',
            'service_date.*' => 'required|date',
            'service_label' => 'required|array',
            // 'service_label.*' => 'filled',
            'amc_product' => 'required|array',
            'amc_product.*' => 'required|exists:products,id',
            'amc_product_note' => 'required|array',
            // 'amc_product_note.*' => 'filled',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('amc_add')) {

                $amc = Amc::create($request->except('attachment'));

                if ($request->has('attachment')) {

                    $attachment = $this->uploadFile($request->attachment, 'images/amc');
                    if ($attachment != false) {
                        $amc->attachment = $attachment;
                    }
                }
                $amc->save();

                $amcStatusLabel = categoryLabel($request->status);
                // Get service status category
                $serviceStatus = Category::where(['key' => 'service_status', 'value' => $amcStatusLabel])->first();

                //Change status parameter of request from amc to service for employee status
                $request->merge([
                    'status' => $serviceStatus->id,
                ]);

                // AMC Service Schedule
                foreach ($request->service_date as $key => $value) {
                    $service = $amc->services()->create([
                        'service_date' => $request->service_date[$key],
                        'service_time' => $request->service_time,
                        'service_label' => $request->service_label[$key],
                        'amc_id' => $amc->id,
                        'status' => ($serviceStatus->id) ? $serviceStatus->id : '',
                        'assign_to' => $request->assign_to,
                        'is_chargeble' => false,
                        'charge' => 0,
                        'remark' => '',
                    ]);

                    // Add each product to service, so can access it in service record as well
                    foreach ($request->amc_product as $key => $value) {
                        $service->products()->create([
                            'service_product' => $request->amc_product[$key],
                            'product_note' => $request->amc_product_note[$key],
                        ]);
                    }
                }

                // AMC Selected Products
                foreach ($request->amc_product as $key => $value) {
                    $amc->products()->create([
                        'amc_product' => $request->amc_product[$key],
                        'product_note' => $request->amc_product_note[$key],
                    ]);
                }
                
                $product = Product::find($request->amc_product[$key]);
                $targetName = $product ? $product->name : 'Unknown Product';

                $sourceName = auth()->id();
                    $users = User::find($sourceName);
                    $sourceId = $users ? $users->first_name : 'Unknown User';
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => AMC::AMC_CREATE_ACTIVITY,
                    'activity_slug' =>Str::of(AMC::AMC_CREATE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => "AMC for product '{$targetName}' created",
                ];
                ActivityLog::log($data);
                $admin = User::role(['admin'])->get();
                $notifications = [
                    [
                        'user_id' => $amc->employee->id , // Change this to the appropriate user ID
                        'target_id' => $amc->id,
                        'updated_by' => $sourceId,
                        'content' => "New AMC Contract Request {$amc->amc_no} Assigned to You.",
                        'is_read' => 0, // Set as unread (0) by default
                        'path' => AMC::AMC_PATH 
                    ],
                    [
                        'user_id' => $amc->client->id , // Change this to the appropriate user ID
                        'target_id' => $amc->id,
                        'updated_by' => $sourceId,
                        'content' => "Your AMC Contract Request {$amc->amc_no} has been received.",
                        'is_read' => 0, // Set as unread (0) by default
                        'path' => AMC::AMC_PATH 
                    ],
                    [
                        'user_id' => $admin->first()->id , // Change this to the appropriate user ID
                        'target_id' => $amc->id,
                        'updated_by' => $sourceId,
                        'content' => "A new AMC Contrcat request {$amc->amc_no} has been received.",
                        'is_read' => 0, // Set as unread (0) by default
                        'path' => AMC::AMC_PATH 
                    ],
                ];
                Notifications::send($notifications);
                $notificationData = [
                    [
                        'deviceToken' => $amc->employee->device_token,
                        'title' => 'AMC Contract Assigned',
                        'body' => "New AMC Contract Request {$amc->amc_no} Assigned to You.",
                    ],
                    [
                        'deviceToken' => $amc->client->device_token,
                        'title' => 'AMC Contract Received',
                        'body' => "Your AMC Contract Request {$amc->amc_no} has been received.",
                    ],
                    [
                        'deviceToken' => $admin->first()->device_token,
                        'title' => 'New AMC Contract Request',
                        'body' => "A new AMC Contract Request {$amc->amc_no} has been received.",
                    ],
                ];
    
                // Trigger event to send FCM notifications
                event(new SendFcmNotification($notificationData));
                $this->_response['data'] = '';
                $this->setResponse(false, 'AMC created successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amcs,id'
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('amc_view')) {

                $amc = Amc::find($request->id);

                return (new AmcResource($amc))->additional(["error" => false, "message" => '']);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amcs,id',
            'customer' => 'required|exists:users,id',
            'date' => 'required|date',
            'contact_person' => 'nullable',
            'status' => 'required|exists:categories,id',
            'assign_to' => 'required|exists:users,id',
            'attachment' => 'filled|mimes:jpg,bmp,png,jpeg,svg,doc,docx,pdf,csv,txt|max:2048',
            // 'amc_detail' => 'required',
            'interval' => 'required|numeric|min:1',
            'number_of_service' => 'required|numeric|min:1',
            'service_date' => 'required|array',
            'service_date.*' => 'required|date',
            'service_label' => 'required|array',
            // 'service_label.*' => 'required',
            'amc_product' => 'required|array',
            'amc_product.*' => 'required|exists:products,id',
            'amc_product_note' => 'filled|array',
            'amc_product_note.*' => 'nullable',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true,  $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('amc_edit')) {

                $amc = Amc::find($request->id);
                $old_employee_status_id = $amc->employee_status;
                if ($request->status != $request->employee_status && auth()->user()->getRole() != 'admin') {
                    $request->request->add(['is_employee_status_approved' => false]);
                } else {
                    $request->request->add(['employee_status' => $request->status, 'is_employee_status_approved' => true]);
                }
                $amc->update($request->except(['attachment']));

                if ($request->has('attachment')) {

                    $attachment = $this->uploadFile($request->attachment, 'images/amc');
                    if ($attachment != false) {
                        $amc->attachment = $attachment;
                    }
                }
                $amc->save();

                // Delete old records
                $amc->services()->delete();

                $amcStatusLabel = categoryLabel($request->status);
                // Get service status category
                $serviceStatus = Category::where(['key' => 'service_status', 'value' => $amcStatusLabel])->first();

                //Change status parameter of request from amc to service for employee status
                $request->merge([
                    'status' => $serviceStatus->id,
                ]);

                // AMC Service Schedule
                foreach ($request->service_date as $key => $value) {
                    $service = $amc->services()->create([
                        'service_date' => $request->service_date[$key],
                        'service_time' => $request->service_time,
                        'service_label' => $request->service_label[$key],

                        'amc_id' => $amc->id,
                        'status' => ($serviceStatus->id) ? $serviceStatus->id : '',
                        'assign_to' => $request->assign_to,
                        'is_chargeble' => false,
                        'charge' => 0,
                        'remark' => '',
                    ]);

                    // Add each product to service, so can access it in service record as well
                    foreach ($request->amc_product as $key => $value) {
                        $service->products()->create([
                            'service_product' => $request->amc_product[$key],
                            'product_note' => $request->amc_product_note[$key],
                        ]);
                    }
                }

                // Delete old records
                $amc->products()->delete();
                // AMC Selected Products
                foreach ($request->amc_product as $key => $value) {
                    $amc->products()->create([
                        'amc_product' => $request->amc_product[$key],
                        'product_note' => $request->amc_product_note[$key],
                    ]);
                }

                $product = Product::find($request->amc_product[$key]);
                $admin = User::role(['admin'])->get();
                $sourceName = auth()->id();
                    $users = User::find($sourceName);
                    $sourceId = $users ? $users->first_name : 'Unknown User';
                $targetName = $product ? $product->name : 'Unknown Product';
                $data = [
                    'updated_by' => $sourceId,
                    'activity' => AMC::AMC_UPDATE_ACTIVITY,
                    'activity_slug' =>Str::of(AMC::AMC_UPDATE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' =>"AMC for product '{$targetName}' updated",
                ];
                ActivityLog::log($data);
                if ($old_employee_status_id != $request->employee_status){
                    if(auth()->user()->getRole() == 'admin') {
                        switch ($request->employee_status) {
                            case 15: // In Progress
                                $notifications = [
                                    [
                                        'user_id' => $amc->client->id,
                                        'target_id' => $amc->id,
                                        'updated_by' => $sourceId,
                                        'content' => "Your AMC Contract Request {$amc->amc_no} is now in progress.",
                                        'is_read' => 0,
                                        'path' => AMC::AMC_PATH
                                    ],
                                    [
                                        'user_id' => $amc->employee->id,
                                        'target_id' => $amc->id,
                                        'updated_by' => $sourceId,
                                        'content' => "AMC Contract in progress status for {$amc->amc_no} has been approved.",
                                        'is_read' => 0,
                                        'path' => AMC::AMC_PATH
                                    ],
                                ];
                                Notifications::send($notifications);
                                $notificationData = [
                                    [
                                        'deviceToken' => $amc->client->device_token,
                                        'title' => 'AMC Contract Status Update',
                                        'body' => "Your AMC Contract Request {$amc->amc_no} is now in progress.",
                                    ],
                                    [
                                        'deviceToken' => $amc->employee->device_token,
                                        'title' => 'AMC Contract Status Update',
                                        'body' => "AMC Contract in progress status for {$amc->amc_no} has been approved.",
                                    ],
                                ];
                                event(new SendFcmNotification($notificationData));
                                break;
                            case 16: // Completed
                                $notifications = [
                                    [
                                        'user_id' => $amc->client->id,
                                        'target_id' => $amc->id,
                                        'updated_by' => $sourceId,
                                        'content' => "Your AMC Contract Request {$amc->amc_no} has been completed.",
                                        'is_read' => 0,
                                        'path' => AMC::AMC_PATH
                                    ],
                                    [
                                        'user_id' => $amc->employee->id,
                                        'target_id' => $amc->id,
                                        'updated_by' => $sourceId,
                                        'content' => "AMC Contract completed status for {$amc->amc_no} has been approved.",
                                        'is_read' => 0,
                                        'path' => AMC::AMC_PATH
                                    ],
                                ];
                                Notifications::send($notifications);
                                $notificationData = [
                                    [
                                        'deviceToken' => $amc->client->device_token,
                                        'title' => 'AMC Contract Status Update',
                                        'body' => "Your AMC Contract Request {$amc->amc_no} has been completed.",
                                    ],
                                    [
                                        'deviceToken' => $amc->employee->device_token,
                                        'title' => 'AMC Contract Status Update',
                                        'body' => "AMC Contract completed status for {$amc->amc_no} has been approved.",
                                    ],
                                ];
                                event(new SendFcmNotification($notificationData));
                                break;
                            // Add more cases for other status changes if needed
                            default:
                                // No action for other status changes
                                break;
                        }
                    }
                    else{
                    switch ($request->employee_status) {
                        case 15: // In Progress
                            $notifications = [
                                [
                                    'user_id' => $admin->first()->id,
                                    'target_id' => $amc->id,
                                    'updated_by' => $sourceId,
                                    'content' => "AMC Contract Request {$amc->amc_no} is now in progress.",
                                    'is_read' => 0,
                                    'path' => AMC::AMC_PATH
                                ]
                            ];
                            Notifications::send($notifications);
                            $notificationData = [
                                [
                                    'deviceToken' => $admin->first()->device_token,
                                    'title' => 'AMC Contract Status Update',
                                    'body' => "AMC Contract Request {$amc->amc_no} is now in progress.",
                                ],
                            ];
                            event(new SendFcmNotification($notificationData));
                            break;
                        case 16: // Completed
                            $notifications = [
                                [
                                    'user_id' => $admin->first()->id,
                                    'target_id' => $amc->id,
                                    'updated_by' => $sourceId,
                                    'content' => "AMC Contract Request {$amc->amc_no} has been completed.",
                                    'is_read' => 0,
                                    'path' => AMC::AMC_PATH
                                ],
                            ];
                            Notifications::send($notifications);
                            $notificationData = [
                                [
                                    'deviceToken' => $admin->first()->device_token,
                                    'title' => 'AMC Contract Status Update',
                                    'body' => "AMC Contract Request {$amc->amc_no} has been completed.",
                                ],
                            ];
                            event(new SendFcmNotification($notificationData));
                            break;
                        // Add more cases for other status changes if needed
                        default:
                            // No action for other status changes
                            break;
                    }
                };
                };
                $this->_response['data'] = '';
                $this->setResponse(false, 'AMC updated successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true,  $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amcs,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('amc_delete')) {

                $amc =  Amc::find($request->id);
                if ($amc) {
                    
                    $sourceName = auth()->id();
                    $users = User::find($sourceName);
                    $sourceId = $users ? $users->first_name : 'Unknown User';
                    $amcProduct = AmcProduct::where('amc_id', $amc->id)->first();
                    $product = $amcProduct ? Product::find($amcProduct->amc_product) : null;
                    $targetName = $product ? $product->name : 'Unknown Product';

                    $activityDetails = "AMC for product '{$targetName}' deleted";
                    
                    $data = [
                        'updated_by' => $sourceId,
                        'activity' => AMC::AMC_DELETE_ACTIVITY,
                        'activity_slug' =>Str::of(AMC::AMC_DELETE_ACTIVITY)->slug('_'),
                        'target_name' => $targetName,
                        'details' => $activityDetails,
                    ];
                    ActivityLog::log($data);
                    $amc->delete();
                    $this->setResponse(false, 'Amc deleted successfully.');
                    return response()->json($this->_response);
                }
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /*Delete multiple AMC Data*/
    public function deleteMultiAmc(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amc_ids' => 'required|array',
            'amc_ids.*' => 'required|exists:amcs,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }
        try {
            if (auth()->user()->getRole() == 'admin' || auth()->user()->hasPermissionTo('amc_delete')) {

                foreach ($request->amc_ids as $id) {
                    $amc =  Amc::find($id);
                    if ($amc) {
                        $amc->delete();
                    }
                }

                $sourceName = auth()->id();
                $users = User::find($sourceName);
                $sourceId = $users ? $users->first_name : 'Unknown User';
                $product = Product::find($request->amc_product[$amc->amc_id]);
                // $targetName = "AMC ID: {$amc->id}";
                $targetName = $product ? $product->name : 'Unknown Product';

                $activityDetails = "AMC for product '{$targetName}' deleted";

                $data = [
                    'updated_by' => $sourceId,
                    'activity' => AMC::AMC_DELETE_ACTIVITY,
                    'activity_slug' =>Str::of(AMC::AMC_DELETE_ACTIVITY)->slug('_'),
                    'target_name' => $targetName,
                    'details' => $activityDetails,
                ];
                ActivityLog::log($data);

                $this->setResponse(false, 'Amc deleted successfully.');
                return response()->json($this->_response);
            } else {
                $data = [
                    'error' => true,
                    'message' => 'Unauthorized',
                    'error_code' => 401
                ];
                return response()->json($data, 401);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function updateStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amcs,id',
            'status' => 'required|exists:categories,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $amc =  Amc::find($request->id);
            if ($amc) {
                // $old_employee_status_is_updated = $amc->is_employee_status_approved;
                $amc->update(['status' => $request->status, 'employee_status' => $request->status]);

                $this->setResponse(false, 'Status updated successfully.');
                return response()->json($this->_response);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function rejectEmployeeStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amcs,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $amc =  Amc::find($request->id);
            if ($amc) {
                // $old_status = $amc->status;
                $amc->update(['is_employee_status_approved' => true]);

                $sourceName = auth()->user()->first_name ?? 'Unknown User';

                // Prepare notifications based on AMC status
                $notifications = [];
                $notificationData = [];
    
                switch ($amc->status) {
                    case 14: // Open
                        $notifications[] = [
                            'user_id' => $amc->client->id,
                            'target_id' => $amc->id,
                            'updated_by' => $sourceName,
                            'content' => "Your AMC Contract open status for {$amc->amc_no} has been rejected.",
                            'is_read' => 0,
                            'path' => AMC::AMC_PATH
                        ];
                        $notificationData[] = [
                            'deviceToken' => $amc->client->device_token,
                            'title' => 'AMC Contract Status Update',
                            'body' => "Your AMC Contract open status for {$amc->amc_no} has been rejected.",
                        ];
                        break;
                    case 15: // In Progress
                        $notifications[] = [
                            'user_id' => $amc->employee->id,
                            'target_id' => $amc->id,
                            'updated_by' => $sourceName,
                            'content' => "AMC Contract in progress status for {$amc->amc_no} has been rejected.",
                            'is_read' => 0,
                            'path' => AMC::AMC_PATH
                        ];
                        $notificationData[] = [
                            'deviceToken' => $amc->employee->device_token,
                            'title' => 'AMC Contract Status Update',
                            'body' => "AMC Contract in progress status for {$amc->amc_no} has been rejected.",
                        ];
                        break;
                    case 16: // Closed
                        $notifications[] = [
                            'user_id' => $amc->employee->id,
                            'target_id' => $amc->id,
                            'updated_by' => $sourceName,
                            'content' => "AMC Contract closed status for {$amc->amc_no} has been rejected.",
                            'is_read' => 0,
                            'path' => AMC::AMC_PATH
                        ];
                        $notificationData[] = [
                            'deviceToken' => $amc->employee->device_token,
                            'title' => 'AMC Contract Status Update',
                            'body' => "AMC Contract closed status for {$amc->amc_no} has been rejected.",
                        ];
                        break;
                    default:
                        // Handle other status codes if needed
                        break;
                }
    
                // Send the prepared notifications
                Notifications::send($notifications);
                event(new SendFcmNotification($notificationData));
                $this->setResponse(false, 'Status rejected successfully.');
                return response()->json($this->_response);
            }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Amc  $amc
     * @return \Illuminate\Http\Response
     */
    public function approveEmployeeStatus(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|exists:amcs,id',
        ]);

        if ($validator->fails()) {
            $this->setResponse(true, $validator->errors()->all());
            return response()->json($this->_response, 400);
        }

        try {
            $amc =  Amc::find($request->id);
            if ($amc) {
                // $old_employee_status_is_updated = $amc->is_employee_status_approved;
                $amc->update(['status' => $amc->employee_status, 'is_employee_status_approved' => true]);
                // Get the source user's name
            $sourceName = auth()->user()->first_name ?? 'Unknown User';

            // Prepare notifications based on AMC status
            $notifications = [];
            $notificationData = [];

            switch ($amc->status) {
                case 14: // Open
                    $notifications[] = [
                        'user_id' => $amc->client->id,
                        'target_id' => $amc->id,
                        'updated_by' => $sourceName,
                        'content' => "Your AMC Contract Request {$amc->amc_no} has been opened.",
                        'is_read' => 0,
                        'path' => AMC::AMC_PATH
                    ];
                    $notificationData[] = [
                        'deviceToken' => $amc->client->device_token,
                        'title' => 'AMC Contract Update',
                        'body' => "Your AMC Contract Request {$amc->amc_no} has been opened.",
                    ];
                    break;
                case 15: // In Progress
                    $notifications[] = [
                        'user_id' => $amc->client->id,
                        'target_id' => $amc->id,
                        'updated_by' => $sourceName,
                        'content' => "Your AMC Contract Request {$amc->amc_no} is now in progress.",
                        'is_read' => 0,
                        'path' => AMC::AMC_PATH
                    ];
                    $notifications[] = [
                        'user_id' => $amc->employee->id,
                        'target_id' => $amc->id,
                        'updated_by' => $sourceName,
                        'content' => "AMC Contract in progress status for {$amc->amc_no} has been approved.",
                        'is_read' => 0,
                        'path' => AMC::AMC_PATH
                    ];
                    $notificationData[] = [
                        'deviceToken' => $amc->client->device_token,
                        'title' => 'AMC Contract Status Update',
                        'body' => "Your AMC Contract Request {$amc->amc_no} is now in progress.",
                    ];
                    $notificationData[] = [
                        'deviceToken' => $amc->employee->device_token,
                        'title' => 'AMC Contract Status Update',
                        'body' => "AMC Contract in progress status for {$amc->amc_no} has been approved.",
                    ];
                    break;
                case 16: // Closed
                    $notifications[] = [
                        'user_id' => $amc->client->id,
                        'target_id' => $amc->id,
                        'updated_by' => $sourceName,
                        'content' => "Your AMC Contract Request {$amc->amc_no} has been closed.",
                        'is_read' => 0,
                        'path' => AMC::AMC_PATH
                    ];
                    $notifications[] = [
                        'user_id' => $amc->employee->id,
                        'target_id' => $amc->id,
                        'updated_by' => $sourceName,
                        'content' => "AMC Contract closed status for {$amc->amc_no} has been approved.",
                        'is_read' => 0,
                        'path' => AMC::AMC_PATH
                    ];
                    $notificationData[] = [
                        'deviceToken' => $amc->client->device_token,
                        'title' => 'AMC Contract Status Update',
                        'body' => "Your AMC Contract Request {$amc->amc_no} has been closed.",
                    ];
                    $notificationData[] = [
                        'deviceToken' => $amc->employee->device_token,
                        'title' => 'AMC Contract Status Update',
                        'body' => "AMC Contract closed status for {$amc->amc_no} has been approved.",
                    ];
                    break;
                default:
                    // Handle other status codes if needed
                    break;
            }

            // Send the prepared notifications
            Notifications::send($notifications);
            event(new SendFcmNotification($notificationData));
                $this->_response['data'] = '';
                $this->setResponse(false, 'Status approved successfully.');
                return response()->json($this->_response);
        }
        } catch (\Exception $e) {
            $this->setResponse(true, $e->getMessage());
            return response()->json($this->_response, 500);
        }
    }
}
